/* vim:set sw=4 ts=8 fileencoding=cp1251:::WINDOWS-1251[] */
/*
 * Copyright(C) 2000-2013  
 *
 *    , 
 *    .
 *
 *        ,
 * ,    ,
 *     ,
 * ,      
 *     
 *      .
 *
 *  ,    , 
 *         
 *   .
 *
 *  -   
 *     .
 */
/*!
 * \file $RCSfile$
 * \version $Revision: 204989 $
 * \date $Date:: 2020-01-15 08:55:47 +0300#$
 * \author $Author: dim $
 *
 * \brief  ()      
 * " CSP"
 *
 * : 
 *      dmntcs -test  [-blocksize <- >] [-numblock <- >] 
 *              [-container < >] -device </dev/drtcsp[0-9]>
 *
 *      dmntcs -create -container < > 
 *
 *      dmntcs -delete -container < > 
 *
 *      dmntcs -makepub -public <   >
 *              -container < > 
 *
 *      dmntcs -encrypt -public <   >
 *              [-blocksize <- >] [-numblock <- >] 
 *              -container < > -device </dev/drtcsp[0-9]> 
 *              -file < > 
 *
 *      dmntcs -decrypt -container < > -file < > 
 *              -device </dev/drtcsp[0-9]>
 *
 *      dmntcs -selftest [-numblock <- >] 
 *              [-container < >] -device </dev/drtcsp[0-9]>
 *
 * :
 *
 *      /dev/drtcsp[01] 
 *                ,   .
 *
 *      /dev/drtcsp[23]
 *             CPHashSessionKey/CPDeriveKey   ,   .
 *
 *      /dev/drtcsp[45]
 *              CPGenKey,    .
 *
 *      /dev/drtcsp[67]
 *            ==  .
 *
 *      /dev/drtcsp[89]
 *            ,   .
 *
 *  :
 *
 *      -     ;
 *      -    ;
 *      -     ;
 *      -    ;
 *      -    ;
 *      -    ;
 *      - #  (<# >/2)
 *      -   ;
 *      - -  ;
 *      -     0;
 */

#define MODULE  "dmntsc"

#include "common.h"
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#ifdef USE_DAEMON
#include <sys/ioctl.h>
#endif
#include <netinet/in.h>

#include "getopt.h"
#include "drtcsp_io.h"
#include "Crypt.h"

static HCRYPTMODULE csp;
static VTABLEPROVSTRUC VTABLE2001 = { 3, NULL, NULL, PROV_GOST_2001_DH,
    NULL, 0, 0 }; 
static VTABLEPROVSTRUC *CurVTable = &VTABLE2001;

static void
usage(void)
{
  fprintf(stderr, "%s\n", 
	  "Usage:\n"
	  "   dmntcs -test  [-blocksize <- >] [-numblock <- >]\n"
	  "           [-container < >] -device </dev/drtcsp[0-9]>\n"
	  "\n"
	  "   dmntcs -create -container < > \n"
	  "\n"
	  "   dmntcs -delete -container < > \n"
	  "\n"
	  "   dmntcs -makepub -public <   > \n"
	  "           -container < > \n"
	  "\n"
	  "   dmntcs -encrypt [-public <   >]\n"
	  "           [-blocksize <- >] [-numblock <- >] \n"
	  "           -container < > -device </dev/drtcsp[0-9]> \n"
	  "           -file < > \n"
	  "\n"
	  "   dmntcs -decrypt -container < > -file < > \n"
	  "           -device </dev/drtcsp[0-9]>\n"
	  "\n"
	  "   dmntcs -selftest [-numblock <- >]\n"
	  "           [-container < >] -device </dev/drtcsp[0-9]>\n"
	  "\n"
	  __DATE__ " " __TIME__ "\n"
	  );
}

struct ACTION_ARGS;

typedef int FACTION(const struct ACTION_ARGS *pargs);
typedef FACTION *PFACTION;

typedef struct ACTION_ARGS{
  FACTION     *pfAction;
  const
  char        *lpszContainerName;
  const
  char        *lpszDeviceName;
  const
  char        *lpszFileName;
  const
  char        *lpszPublicName;
  size_t      siBlockSize;
  size_t      siNumBlocks;
} ACTION_ARGS, *PACTION_ARGS;

static FACTION fa_test;
static FACTION fa_create;
static FACTION fa_delete;
static FACTION fa_makepub;
static FACTION fa_encrypt;
static FACTION fa_decrypt;
static FACTION fa_selftest;

#define MAX_THREADS     ((size_t)10)

int
main(int ac, char *av[]) {
  ACTION_ARGS acts[MAX_THREADS];
  ACTION_ARGS *pa;
  int         retval = 1;
  int         c;

  static struct option long_options[] = {
    {"help",        no_argument,        NULL,   'h'},
        
    {"test",        no_argument,        NULL,   'T'},
    {"create",      no_argument,        NULL,   'C'},
    {"delete",      no_argument,        NULL,   'R'},
    {"makepub",     no_argument,        NULL,   'M'},
    {"encrypt",     no_argument,        NULL,   'E'},
    {"decrypt",     no_argument,        NULL,   'D'},
    {"selftest",    no_argument,        NULL,   'S'},
        
    {"container",   required_argument,  NULL,   'c'},
    {"public",      required_argument,  NULL,   'p'},
    {"file",        required_argument,  NULL,   'f'},
    {"device",      required_argument,  NULL,   'd'},
    {"blocksize",   required_argument,  NULL,   'b'},
    {"numblocks",   required_argument,  NULL,   'n'},
    {0, 0, 0, 0}
  };

  pa = acts;
  pa--;
  while(EOF != (c = getopt_long_only(ac, av, "", long_options, NULL)) &&
	pa < &acts[sizeof(acts)/sizeof(acts[0]) - 1]){
    switch (c) {
    case 'T':
      pa++;
      pa->pfAction = fa_test;
      pa->lpszContainerName = "\\\\.\\hdimage\\dmntsc";
      pa->lpszPublicName = NULL;
      pa->lpszFileName = NULL;
      pa->lpszDeviceName = "/dev/drtcsp0";
      pa->siBlockSize = 1024;
      pa->siNumBlocks = 1;
      continue;
    case 'C':
      pa++;
      pa->pfAction = fa_create;
      pa->lpszContainerName = "\\\\.\\hdimage\\dmntsc";
      pa->lpszPublicName = NULL;
      pa->lpszFileName = NULL;
      pa->lpszDeviceName = "/dev/drtcsp0";
      pa->siBlockSize = 0;
      pa->siNumBlocks = 0;
      continue;
    case 'R':
      pa++;
      pa->pfAction = fa_delete;
      pa->lpszContainerName = "\\\\.\\hdimage\\dmntsc";
      pa->lpszPublicName = NULL;
      pa->lpszFileName = NULL;
      pa->lpszDeviceName = "/dev/drtcsp0";
      pa->siBlockSize = 0;
      pa->siNumBlocks = 0;
      continue;
    case 'M':
      pa++;
      pa->pfAction = fa_makepub;
      pa->lpszContainerName = "\\\\.\\hdimage\\dmntsc";
      pa->lpszPublicName = "dmntsc.pub";
      pa->lpszFileName = NULL;
      pa->lpszDeviceName = "/dev/drtcsp0";
      pa->siBlockSize = 0;
      pa->siNumBlocks = 0;
      continue;
    case 'E':
      pa++;
      pa->pfAction = fa_encrypt;
      pa->lpszContainerName = "\\\\.\\hdimage\\dmntsc";
      pa->lpszPublicName = "dmntsc.pub";
      pa->lpszFileName = "encrypts0s.enc";
      pa->lpszDeviceName = "/dev/drtcsp0";
      pa->siBlockSize = 1024;
      pa->siNumBlocks = 1;
      continue;
    case 'D':
      pa++;
      pa->pfAction = fa_decrypt;
      pa->lpszContainerName = "\\\\.\\hdimage\\dmntsc";
      pa->lpszPublicName = "dmntsc.pub"; /* MUST be NULL */
      pa->lpszFileName = "encrypts0s.enc";
      pa->lpszDeviceName = "/dev/drtcsp0";
      pa->siBlockSize = 0;
      pa->siNumBlocks = 0;
      continue;
    case 'S':
      pa++;
      pa->pfAction = fa_selftest;
      pa->lpszContainerName = "\\\\.\\hdimage\\dmntsc";
      pa->lpszPublicName = NULL;
      pa->lpszFileName = NULL;
      pa->lpszDeviceName = "/dev/drtcsp0";
      pa->siBlockSize = 0;
      pa->siNumBlocks = 1;
      continue;
    }
                
    if (pa < acts) {
      usage();
      goto err_ret;
    }

    switch (c) {
    case 'c':
      pa->lpszContainerName = optarg;
      continue;
    case 'p':
      pa->lpszPublicName = optarg;
      continue;
    case 'f':
      pa->lpszFileName = optarg;
      continue;
    case 'd':
      pa->lpszDeviceName = optarg;
      continue;
    case 'b':
      pa->siBlockSize = atol (optarg);
      continue;
    case 'n':
      pa->siNumBlocks = atol (optarg);
      continue;
       
    case 'h':
    default:
      usage();
      goto err_ret;
    }
  }

  if(pa < &acts[0]){    
    usage();
    goto err_ret;
  }
    
  pa++;    
  pa->pfAction = NULL;
  if(CPC2CryptCreateProvider(&csp, NULL) || !csp){
    printf("Failed to load CSP.\n");
    return 1;
  }  

  for(pa = acts; NULL != pa->pfAction; pa++){
    if(0 != (retval = (*pa->pfAction)(pa))){
      break;
    }
  }

 err_ret:
  if (csp) csp->DestroyProvider(csp);

  printf(MODULE"!main: retval=%d\n", retval);
    
  return retval;
}

#ifdef USE_DAEMON
static int
daemon_handshake(const ACTION_ARGS *pargs, HCRYPTPROV hProv, int *pdrv,
		 HCRYPTKEY *phDrvKey)
{
  HCRYPTHASH  hDrvHash = 0;
  DRTCSP_DRIVER_KEY   tmp;
  DWORD   dwcbData;
  int retval=2;

  if ((*pdrv = open(pargs->lpszDeviceName, O_RDWR)) < 0) {
    perror(pargs->lpszDeviceName);
    goto err_ret;
  }
  
  tmp.size = (uint16_t)sizeof(tmp);
  tmp.version = Version1_2;
  tmp.magic = Drtcsp_RN_Daemon_Magic;
  dwcbData = (uint8_t)sizeof(tmp.bData);
  
  if(csp->GetProvParam(csp, hProv, PP_RANDOM, tmp.bData, &dwcbData, 0)) {
    printf(MODULE "!daemon_handshake:%d: GetProvParam\n", __LINE__);
    goto err_ret;
  }

  tmp.cbData = dwcbData;
    
  if(csp->CreateHash(csp, hProv, CALG_GR3411, 0, 0, &hDrvHash)) {
    printf(MODULE "!daemon_handshake:%d: CreateHash\n", __LINE__);
    goto err_ret;
  }
  
  if(csp->HashData(csp, hProv, hDrvHash, (BYTE *)"DriverDaemon", (DWORD)sizeof("DriverDaemon"), 0)) {
    printf(MODULE "!daemon_handshake:%d: HashData(...'DriverDaemon'\n", __LINE__);
    goto err_ret;
  }

  if(csp->HashData(csp, hProv, hDrvHash, tmp.bData, tmp.cbData, 0)) {
    printf(MODULE "!daemon_handshake:%d: HashData(...tmp.bData\n", __LINE__);
    goto err_ret;
  }
  
  if(ioctl(*pdrv, DRTCSP_DRIVER_RNG, &tmp) < 0) {
    printf("%lx\n", (unsigned long)&tmp);
    perror(MODULE "!daemon_handshake:ioctl(drv, DRTCSP_DRIVER_RNG, &tmp)");
    goto err_ret;
  }

  if(tmp.size != sizeof(tmp) || tmp.version != Version1_2 || tmp.magic != Drtcsp_RN_Driver_Magic) {
    printf(MODULE "!daemon_handshake:%d: Drtcsp_RN_Driver_Magic: corrupt\n", __LINE__);
    goto err_ret;
  }

  if(csp->HashData(csp, hProv, hDrvHash, tmp.bData, tmp.cbData, 0)) {
    printf(MODULE "!daemon_handshake:%d: HashData(...tmp.bData\n", __LINE__);
    goto err_ret;
  }

  if(csp->DeriveKey(csp, hProv, CALG_G28147, hDrvHash, 0, phDrvKey)) {
    printf(MODULE "!daemon_handshake:%d: DeriveKey\n", __LINE__);
    goto err_ret;
  }

  if(csp->DestroyHash(csp, hProv, hDrvHash)) {
    printf(MODULE "!daemon_handshake:%d: DestroyHash\n", __LINE__);
    goto err_ret;
  }
  hDrvHash = 0;

  retval=0;
err_ret:
  if(hDrvHash) csp->DestroyHash(csp, hProv, hDrvHash);
  memset(&tmp, 0, sizeof(tmp));
  
  return retval;
}

static int
daemon_set_session_key(HCRYPTPROV hProv, int drv, HCRYPTKEY hSessionKey,
		       HCRYPTKEY hDrvKey)
{
  DRTCSP_DRIVER_KEY   tmp;
  DWORD   dwcbData;
  DWORD   dwAlgId;
  int retval=3;

  tmp.size = (uint16_t)sizeof(tmp);
  tmp.version = Version1_2;
  tmp.magic = Drtcsp_KE_Session_Magic;
  tmp.cbData = (uint8_t)sizeof(tmp.bData);
  dwcbData = tmp.cbData;
  
  dwAlgId = CALG_SIMPLE_EXPORT;
  if (csp->SetKeyParam(csp, hProv, hDrvKey, KP_ALGID, (BYTE *)&dwAlgId, 0)) {
    printf(MODULE "!%s:%d: SetKeyParam(...hDrvKey\n", __func__, __LINE__);
    goto err_ret;
  }
  
  if(csp->ExportKey(csp, hProv, hSessionKey, hDrvKey, SIMPLEBLOB, 0, tmp.bData, &dwcbData)) {
    printf(MODULE "!%s:%d: ExportKey(...hSessionKey)\n", __func__, __LINE__);
    goto err_ret;
  }

  tmp.cbData = dwcbData;
    
  if(ioctl(drv, DRTCSP_SESSION_KEY, &tmp) < 0) {
    perror(MODULE "!%s:ioctl(drv, DRTCSP_SESSION_KEY, __func__, &tmp)");
    goto err_ret;
  }

  if(tmp.size != sizeof(tmp) || tmp.version != Version1_2 || tmp.magic != Drtcsp_KE_OKSetSK_Magic) {
    printf(MODULE "!%s:%d: Drtcsp_KE_OKSetSK_Magic: corrupt\n", __func__, __LINE__);
    goto err_ret;
  }

  retval=0;
err_ret:

  memset(&tmp, 0, sizeof(tmp));

  return retval;
}

static int
daemon_kernel_selftest(int drv)
{
  DRTCSP_DRIVER_KEY   tmp;
  int retval=4;

  tmp.size = (uint16_t)sizeof(tmp);
  tmp.version = Version1_2;
  tmp.magic = Drtcsp_ST_Daemon_Magic;
  tmp.cbData = (uint8_t)sizeof(tmp.bData);
  
  if(ioctl(drv, DRTCSP_SELFTEST, &tmp) < 0) {
    perror(MODULE "!%s:ioctl(drv, DRTCSP_SESSION_KEY, __func__, &tmp)");
    goto err_ret;
  }

  if(tmp.size != sizeof(tmp) || tmp.version != Version1_2 || tmp.magic != Drtcsp_ST_Driver_Magic) {
    printf(MODULE "!%s:%d: Drtcsp_ST_Driver_Magic: corrupt\n", __func__, __LINE__);
    goto err_ret;
  }

  retval=0;
err_ret:

  memset(&tmp, 0, sizeof(tmp));

  return retval;
}
#endif /* USE_DAEMON */

static int
fa_test(const ACTION_ARGS *pargs) {
  int drv=-1;
  HCRYPTHASH  hProv = 0;
  HCRYPTKEY   hDrvKey = 0;
  HCRYPTKEY   hSessionKey = 0;
  char    buf[4096];
  int     retval=1;

  char    *pszContainer = (pargs->lpszContainerName ? 
                                strdup(pargs->lpszContainerName) : NULL);

  printf(MODULE
        "!fa_test:%d: -container %s -device %s "
        "-blocksize %u -numblocks %u\n", __LINE__,
        (pargs->lpszContainerName ? pargs->lpszContainerName : "<NULL>"),
        (pargs->lpszDeviceName ? pargs->lpszDeviceName : "<NULL>"),
        (unsigned)pargs->siBlockSize, (unsigned)pargs->siNumBlocks);
// Both numblocks and blocksize are guaranteed to me small	
        
  if (!pargs->lpszDeviceName || pargs->siBlockSize > sizeof(buf) ||
        pargs->siNumBlocks > 1000000){
        printf(MODULE "!fa_test:%d: Bad args\n", __LINE__);
        goto err_ret;
  }
    
  if (csp->AcquireContext(csp, &hProv, pszContainer, 0, CurVTable)) {
        printf(MODULE "!fa_test:%d: AcquireContext\n", __LINE__);
        goto err_ret;
  }
 
#ifdef USE_DAEMON
  int nest_retval;
  nest_retval=daemon_handshake(pargs, hProv, &drv, &hDrvKey);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */

  if (csp->GenKey(csp, hProv, CALG_G28147, CRYPT_EXPORTABLE, &hSessionKey)) {
        printf(MODULE "!fa_test:%d: GenKey(...&hSessionKey)\n", __LINE__);
        goto err_ret;
  }

#ifdef USE_DAEMON
  nest_retval=daemon_set_session_key(hProv, drv, hSessionKey, hDrvKey);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */

  for (unsigned int i = 0; i < pargs->siNumBlocks; i++) {
    if (pargs->siBlockSize > sizeof(buf)) {
      printf(MODULE "!fa_test:%d: pargs->siBlockSize > sizeof(buf)\n",__LINE__);
      goto err_ret;
    }
#ifdef USE_DAEMON
    if (read(drv, buf, pargs->siBlockSize) != (ssize_t)pargs->siBlockSize) {
      perror(MODULE "!fa_test:read(drv, buf, pargs->siBlockSize)");
      goto err_ret;
    }
      
#if 0
    if (lseek(drv, -(off_t)pargs->siBlockSize, SEEK_CUR) < 0) {
      perror(MODULE "!fa_test:lseek(drv, -pargs->siBlockSize, SEEK_CUR)");
      goto err_ret;
    }
#endif
      
    if (pargs->siBlockSize > sizeof(buf)) {
      printf(MODULE "!fa_test:%d: pargs->siBlockSize > sizeof(buf)\n",__LINE__);
      goto err_ret;
    }
    if (write(drv, buf, pargs->siBlockSize) != (ssize_t)pargs->siBlockSize) {
      perror(MODULE "!fa_test:write(drv, buf, pargs->siBlockSize)");
      goto err_ret;
    }
#else /* USE_DAEMON */
    {
      DWORD buf_size=(DWORD)pargs->siBlockSize;
      unsigned char sync[8];
      DWORD sync_len=8;
      unsigned int j;
	
      if(csp->GetKeyParam(csp, hProv,hSessionKey,KP_IV,sync,&sync_len,0)){
        printf(MODULE "!fa_test:%d: !GetKeyParam(...KP_IV...)\n", __LINE__);
	goto err_ret;
      }

      for(j=0;j<buf_size;j++) buf[j]='\0';
      if(csp->Encrypt(csp, hProv,hSessionKey,0,TRUE,0,(unsigned char*)buf,&buf_size,buf_size)){
        printf(MODULE "!fa_test:%d: Encrypt\n", __LINE__);
        goto err_ret;	
      }
	
      buf_size=(DWORD)pargs->siBlockSize;
	
      if(csp->SetKeyParam(csp, hProv, hSessionKey, KP_IV, sync, 0)) {
        printf(MODULE "!fa_decrypt:%d: !SetKeyParam(...KP_IV...)\n", __LINE__);
        goto err_ret;
      }	
      if(csp->Decrypt(csp, hProv,hSessionKey,0,TRUE,0,(unsigned char*)buf,&buf_size)){
        printf(MODULE "!fa_test:%d: Decrypt\n", __LINE__);
        goto err_ret;	
      }
      for(j=0;j<buf_size;j++) if(buf[j]!='\0') break;
      if(j!=buf_size){
        printf(MODULE "!fa_test:%d: buffer is not null\n", __LINE__);
        goto err_ret;	
      }
    }
#endif /* USE_DAEMON */
  }

  memset(&buf, 0, sizeof(buf));

  printf(MODULE "!fa_test:OK\n");
  retval = 0;
    
err_ret:;

  if (hSessionKey) csp->DestroyKey(csp, hProv, hSessionKey);
  if (hDrvKey) csp->DestroyKey(csp, hProv, hDrvKey);
  if (hProv) csp->ReleaseContext(csp, hProv, 0);
  free(pszContainer);
    
  memset(&buf, 0, sizeof(buf));

  if(drv!=-1) close(drv);

  return retval;
}

static 
int fa_create(const ACTION_ARGS *pargs) {
  HCRYPTPROV hProv=0;
  HCRYPTKEY hKey=0;
  int retval=1;
  char *pszContainer=(pargs->lpszContainerName?strdup(pargs->lpszContainerName):NULL);

  printf(MODULE
	 "!fa_create:%d: -container %s\n", __LINE__,
	 (pargs->lpszContainerName ? pargs->lpszContainerName : "<NULL>"));
  
  if(csp->AcquireContext(csp, &hProv,pszContainer,CRYPT_NEWKEYSET,CurVTable)){
    printf(MODULE "!fa_create:%d: AcquireContext\n", __LINE__);
    goto err_ret;
  }
  if ( csp->SetProvParam(csp, hProv, PP_KEYEXCHANGE_PIN,(BYTE *)"",0)) {
        printf(MODULE "!fa_create:%d: SetProvParam\n", __LINE__);
        goto err_ret;
  }

  if(csp->GenKey(csp, hProv,AT_KEYEXCHANGE,CRYPT_EXPORTABLE|(512<<16),&hKey)){
    printf(MODULE "!fa_create:%d: GenKey(...&hKey)\n", __LINE__);
    goto err_ret;
  }

  printf(MODULE "!fa_create:OK\n");
  retval=0;
 err_ret:
  if(hKey) csp->DestroyKey(csp, hProv, hKey);
  if(hProv) csp->ReleaseContext(csp, hProv, 0);
  free(pszContainer);

  return retval;
}

static 
int fa_delete(const ACTION_ARGS *pargs) {
  HCRYPTPROV hProv=0;
  int retval=1;
  char *pszContainer=(pargs->lpszContainerName?strdup(pargs->lpszContainerName):NULL);

  printf(MODULE
	 "!fa_delete:%d: -container %s\n", __LINE__,
	 (pargs->lpszContainerName ? pargs->lpszContainerName : "<NULL>"));
  
  if(csp->AcquireContext(csp, &hProv,pszContainer,CRYPT_DELETEKEYSET,CurVTable)){
    printf(MODULE "!fa_delete:%d: AcquireContext\n", __LINE__);
    goto err_ret;
  }

  printf(MODULE "!fa_delete:OK\n");
  retval=0;
 err_ret:
  if(hProv) csp->ReleaseContext(csp, hProv, 0);
  free(pszContainer);

  return retval;
}

static 
int fa_makepub(const ACTION_ARGS *pargs) {
  HCRYPTPROV hProv=0;
  HCRYPTKEY hKey=0;
  int retval=1;
  BYTE export_blob[1024];
  DWORD export_blob_len=sizeof(export_blob);
  int fd=-1;
  char *pszContainer=(pargs->lpszContainerName?strdup(pargs->lpszContainerName):NULL);


  printf(MODULE
	 "!fa_makepub:%d: -container %s -public %s\n", __LINE__,
	 (pargs->lpszContainerName ? pargs->lpszContainerName : "<NULL>"),
	 (pargs->lpszPublicName ? pargs->lpszPublicName : "<NULL>"));
  
  if(csp->AcquireContext(csp, &hProv,pszContainer,0,CurVTable)){
    printf(MODULE "!fa_makepub:%d: AcquireContext\n", __LINE__);
    goto err_ret;
  }

  if(csp->GetUserKey(csp, hProv, AT_KEYEXCHANGE, &hKey)){
    printf(MODULE "!fa_makepub:%d: GetUserKey\n", __LINE__);
    goto err_ret;
  }

  if(csp->ExportKey(csp, hProv,hKey,0,PUBLICKEYBLOB,0,export_blob,&export_blob_len)){
    printf(MODULE "!fa_makepub:%d: ExportKey(...hKey)\n", __LINE__);
    goto err_ret;
  }
  
  if((fd=open(pargs->lpszPublicName,O_WRONLY|O_TRUNC|O_CREAT,0700))==-1){
    perror(pargs->lpszPublicName);
    goto err_ret;
  }
  if(write(fd,export_blob, export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszPublicName);
    goto err_ret;
  }

  printf(MODULE "!fa_makepub:OK\n");
  retval=0;
err_ret:

  if(hKey) csp->DestroyKey(csp, hProv, hKey);
  if(hProv) csp->ReleaseContext(csp, hProv, 0);
  free(pszContainer);
  if(fd!=-1) close(fd);

  return retval;
}

static 
int fa_encrypt(const ACTION_ARGS *pargs) {
  HCRYPTHASH  hProv = 0;
  HCRYPTKEY hKey=0;
  HCRYPTKEY hExchKey=0;
  HCRYPTKEY   hDrvKey = 0;
  HCRYPTKEY   hSessionKey = 0;
  BYTE    buf[4096];
  BYTE export_blob[1024];
  DWORD export_blob_len=1024, no_len;
  int     retval=1;
  int fd=-1, fd1=-1;
  char *pszContainer=(pargs->lpszContainerName?strdup(pargs->lpszContainerName):NULL);
  HRESULT hresult;
#ifdef USE_DAEMON
  int drv = -1;
  int nest_retval = 0;
#endif //USE_DAEMON
  
  printf(MODULE
	 "!fa_encrypt:%d: -container %s -device %s "
	 "-blocksize %u -numblocks %u -public %s -file %s\n", __LINE__,
	 (pargs->lpszContainerName ? pargs->lpszContainerName : "<NULL>"),
	 (pargs->lpszDeviceName ? pargs->lpszDeviceName : "<NULL>"),
	 (unsigned)pargs->siBlockSize,(unsigned)pargs->siNumBlocks,
	 (pargs->lpszPublicName ? pargs->lpszPublicName : "<NULL>"),
	 (pargs->lpszFileName ? pargs->lpszFileName : "<NULL>"));
  
  if (!pargs->lpszDeviceName || !pargs->lpszPublicName ||
      !pargs->lpszFileName || pargs->siBlockSize > sizeof(buf) ||
      pargs->siNumBlocks > 1000000){
    printf(MODULE "!fa_encrypt:%d: Bad args\n", __LINE__);
    goto err_ret;
  }
    
  hresult = csp->AcquireContext(csp, &hProv, pszContainer, 0, CurVTable);
  if(hresult) {
    printf(MODULE "!fa_encrypt:%d: AcquireContext: %x\n", __LINE__, hresult);
    goto err_ret;
  }
  
#ifdef USE_DAEMON
  nest_retval=daemon_handshake(pargs, hProv, &drv, &hDrvKey);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */

  if(csp->GetUserKey(csp, hProv, AT_KEYEXCHANGE, &hKey)){
    printf(MODULE "!fa_encrypt:%d: GetUserKey\n", __LINE__);
    goto err_ret;
  }

  export_blob_len=1024;
  if(csp->ExportKey(csp, hProv,hKey,0,PUBLICKEYBLOB,0,export_blob,(DWORD*)&export_blob_len)){
    printf(MODULE "!fa_encrypt:%d: ExportKey(...hKey)\n", __LINE__);
    goto err_ret;
  }
  
  /* Begin writing out file */
  if((fd=open(pargs->lpszFileName,O_WRONLY|O_TRUNC|O_CREAT,0700))==-1){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  /* Write own public key */
  no_len=htonl(export_blob_len);
  if(write(fd,&no_len,4)!=4){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  if(write(fd,export_blob,export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  if((fd1=open(pargs->lpszPublicName,O_RDONLY))==-1){
    perror(pargs->lpszPublicName);
    goto err_ret;
  }
  export_blob_len=(DWORD)read(fd1, export_blob, 1024);
  close(fd1);

  /* Write her public key */
  no_len=htonl(export_blob_len);
  if(write(fd,&no_len,4)!=4){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  if(write(fd,export_blob,export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  /* Generate session key */
  if(csp->GenKey(csp, hProv, CALG_G28147, CRYPT_EXPORTABLE, &hSessionKey)) {
    printf(MODULE "!fa_encrypt:%d: GenKey(...&hSessionKey)\n", __LINE__);
    goto err_ret;
  }
  if(csp->ImportKey(csp, hProv, export_blob, (DWORD)export_blob_len, hKey, 0, &hExchKey)) {
    printf(MODULE "!fa_encrypt:%d: ImportKey(...export_blob)\n", __LINE__);
    goto err_ret;
  }
  
  if(csp->ExportKey(csp, hProv, hSessionKey, hExchKey, SIMPLEBLOB, 0, export_blob, (DWORD*)&export_blob_len)){
    printf(MODULE "!fa_encrypt:%d: ExportKey\n", __LINE__);
    goto err_ret;
  }

  /* Write session key */
  no_len=htonl(export_blob_len);
  if(write(fd,&no_len,4)!=4){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  if(write(fd,export_blob,export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszFileName);
    goto err_ret;
  }
#ifndef USE_DAEMON
  /* Write synchro */
  export_blob_len = 8;
  if (csp->GetKeyParam(csp, hProv, hSessionKey, KP_IV, export_blob, &export_blob_len, 0)) {
    printf(MODULE "!fa_encrypt:%d: !GetKeyParam(...KP_IV...)\n", __LINE__);
    goto err_ret;
  }
  no_len=htonl(export_blob_len);
  if(write(fd,&no_len,4)!=4){
    perror(pargs->lpszFileName);
    goto err_ret;
  }
  if(write(fd,export_blob,export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszFileName);
    goto err_ret;
  }
#endif /* USE_DAEMON */

  /* Write algorithm's id */
  no_len=htonl(0);
  if(write(fd,&no_len,4)!=4){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  /* Write test block len */
  no_len=htonl((DWORD)pargs->siBlockSize); /* Check if it does not lead to troubles */
  if(write(fd,&no_len,4)!=4){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

  /* Write test blocks quantity */
  no_len=htonl((DWORD)pargs->siNumBlocks);/* Check if it does not lead to troubles */
 
  if(write(fd,&no_len,4)!=4){
    perror(pargs->lpszFileName);
    goto err_ret;
  }

#ifdef USE_DAEMON
  nest_retval=daemon_set_session_key(hProv, drv, hSessionKey, hDrvKey);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */

  for (unsigned int i = 0; i < pargs->siNumBlocks; i++) {
#ifdef USE_DAEMON
    if (read(drv, buf, pargs->siBlockSize) != (ssize_t)pargs->siBlockSize) {
      perror(MODULE "!fa_encrypt:read(drv, buf, pargs->siBlockSize)");
      goto err_ret;
    }
#else /* USE_DAEMON */
    {
      DWORD buf_size=(DWORD)pargs->siBlockSize;

      for(unsigned int j=0;j<buf_size;j++) buf[j]='\0';
      if(csp->Encrypt(csp, hProv,hSessionKey,0,TRUE,0,buf,&buf_size,buf_size)){
	printf(MODULE "!fa_encrypt:%d: Encrypt\n", __LINE__);
	goto err_ret;	
      }
    }
#endif /* USE_DAEMON */
    
    /* Write test block */
    if(write(fd,buf,pargs->siBlockSize)!=(ssize_t)pargs->siBlockSize){
      perror(pargs->lpszFileName);
      goto err_ret;
    }
  }
  
  memset(&buf, 0, sizeof(buf));
  
  printf(MODULE "!fa_encrypt:OK\n");
  retval = 0;
  
 err_ret:
  
  if(hSessionKey) csp->DestroyKey(csp, hProv, hSessionKey);
  if(hDrvKey) csp->DestroyKey(csp, hProv, hDrvKey);
  if(fd!=-1) close(fd);
  if(hProv) csp->ReleaseContext(csp, hProv, 0);
  free(pszContainer);
  
  memset(&buf, 0, sizeof(buf));
#ifdef USE_DAEMON
  close(drv);
#endif /* USE_DAEMON */
  
  return retval;
}

static 
int fa_decrypt(const ACTION_ARGS *pargs) {
  int drv=-1;
  HCRYPTHASH  hProv = 0;
  HCRYPTKEY hKey=0;
  HCRYPTKEY hExchKey=0;
  HCRYPTKEY   hDrvKey = 0;
  HCRYPTKEY   hSessionKey = 0;
  BYTE    buf[4096];
  BYTE export_blob[1024], our_blob[1024];
  DWORD export_blob_len=1024, our_blob_len=1024, no_len;
  size_t block_size, blocks_num;
  int     retval=1;
  int fd;
  char *pszContainer=(pargs->lpszContainerName?strdup(pargs->lpszContainerName):NULL);

  printf(MODULE
	 "!fa_decrypt:%d: -container %s -device %s "
	 "-blocksize %u -numblocks %u -public %s -file %s\n", __LINE__,
	 (pargs->lpszContainerName ? pargs->lpszContainerName : "<NULL>"),
	 (pargs->lpszDeviceName ? pargs->lpszDeviceName : "<NULL>"),
	 (unsigned)pargs->siBlockSize,(unsigned)pargs->siNumBlocks,
	 (pargs->lpszPublicName ? pargs->lpszPublicName : "<NULL>"),
	 (pargs->lpszFileName ? pargs->lpszFileName : "<NULL>"));
  
  if (!pargs->lpszDeviceName || !pargs->lpszPublicName ||
      !pargs->lpszFileName || pargs->siBlockSize > sizeof(buf) ||
      pargs->siNumBlocks > 1000000){
    printf(MODULE "!fa_decrypt:%d: Bad args\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }
    
  if (csp->AcquireContext(csp, &hProv, pszContainer, 0, CurVTable)) {
    printf(MODULE "!fa_decrypt:%d: AcquireContext\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }
  
#ifdef USE_DAEMON
  int nest_retval;
  nest_retval=daemon_handshake(pargs, hProv, &drv, &hDrvKey);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */
  
  if(csp->GetUserKey(csp, hProv, AT_KEYEXCHANGE, &hKey)){
    printf(MODULE "!fa_decrypt:%d: GetUserKey\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }

  /* Read our public key */
  if((fd=open(pargs->lpszPublicName,O_RDONLY))==-1){
    perror(pargs->lpszPublicName);
    retval = __LINE__;
    goto err_ret;
  }
  our_blob_len=(DWORD)read(fd, our_blob, 1024);
  close(fd);

  /* Begin reading file */
  if((fd=open(pargs->lpszFileName,O_RDONLY))==-1){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  
  /* Read sender's public key */
  if(read(fd, &no_len, 4)!=4){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  export_blob_len=ntohl(no_len);
  if(read(fd, export_blob, export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  if(csp->ImportKey(csp, hProv, export_blob, (DWORD)export_blob_len, hKey, 0, &hExchKey)){
    printf(MODULE "!fa_decrypt:%d: ImportKey(...&hExchKey)\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }

  /* Check if receiver's key is ours */
  if(read(fd, &no_len, 4)!=4){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  if(ntohl(no_len)!=our_blob_len){
    printf(MODULE "!fa_decrypt:%d: blob_len != our_len\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }
  export_blob_len=(DWORD)read(fd, export_blob, our_blob_len);
  if(export_blob_len!=our_blob_len){
    printf(MODULE "!fa_decrypt:%d: blob_len != our_len\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }
  if(memcmp(export_blob, our_blob, our_blob_len)){
    printf(MODULE "!fa_decrypt:%d: public key isn't ours\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }

  /* Read session key */
  if(read(fd, &no_len, 4)!=4){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  export_blob_len=ntohl(no_len);
  if(read(fd, export_blob, export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  if(csp->ImportKey(csp, hProv, export_blob, (DWORD)export_blob_len, hExchKey, CRYPT_EXPORTABLE, &hSessionKey)){
    printf(MODULE "!fa_decrypt:%d: ImportKey(...&hSessionKey)\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }

#ifndef USE_DAEMON
  /* Read synchro */
  if(read(fd, &no_len, 4)!=4){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  export_blob_len=ntohl(no_len);
  if(read(fd, export_blob, export_blob_len)!=(ssize_t)export_blob_len){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }

  if(csp->SetKeyParam(csp, hProv, hSessionKey, KP_IV, export_blob, 0)) {
    printf(MODULE "!fa_decrypt:%d: !SetKeyParam(...KP_IV...)\n", __LINE__);
    retval = __LINE__;
    goto err_ret;
  }
#endif /* USE_DAEMON */

#ifdef USE_DAEMON
  nest_retval=daemon_set_session_key(hProv, drv, hSessionKey, hDrvKey);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */

  
  /* Read algorithm's id */
  if(read(fd, &no_len, 4)!=4){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }

  /* Read test block len */
  if(read(fd, &no_len, 4)!=4){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  block_size=ntohl(no_len);

  /* Read test blocks quantity */
  if(read(fd, &no_len, 4)!=4){
    perror(pargs->lpszFileName);
    retval = __LINE__;
    goto err_ret;
  }
  blocks_num=ntohl(no_len);

  for (unsigned int i = 0; i < blocks_num; i++) {
ssize_t a;
    if((a=read(fd, buf, block_size)) != (ssize_t)block_size) {
      perror(pargs->lpszFileName);
      retval = __LINE__;
      goto err_ret;
    }
#ifdef USE_DAEMON    
    if(write(drv, buf, block_size) != (ssize_t)block_size) {
      perror(MODULE "!fa_decrypt:write(drv, buf, block_size)");
      retval = __LINE__;
      goto err_ret;
    }
#else /* USE_DAEMON */
    {
      DWORD buf_size=(DWORD)block_size;
      unsigned int j;

      if(csp->Decrypt(csp, hProv,hSessionKey,0,TRUE,0,buf,&buf_size)){
	printf(MODULE "!fa_decrypt:%d: Decrypt\n", __LINE__);
	retval = __LINE__;
	goto err_ret;	
      }
      for(j=0;j<buf_size;j++) if(buf[j]!='\0') break;
      if(j!=buf_size){
	printf(MODULE "!fa_decrypt:%d: buffer is not null\n", __LINE__);
	retval = __LINE__;
	goto err_ret;	
      }
    }
#endif /* USE_DAEMON */
  }  

  memset(&buf, 0, sizeof(buf));
  
  printf(MODULE "!fa_decrypt:OK\n");
  retval = 0;
  
 err_ret:
  
  if (hSessionKey) csp->DestroyKey(csp, hProv, hSessionKey);
  if (hDrvKey) csp->DestroyKey(csp, hProv, hDrvKey);
  if (hProv) csp->ReleaseContext(csp, hProv, 0);
  free(pszContainer);
  memset(&buf, 0, sizeof(buf));
  if(drv!=-1) close(drv);
  
  return retval;
}

static int
fa_selftest(const ACTION_ARGS *pargs) {
  int drv=-1;
  HCRYPTHASH  hProv = 0;
  HCRYPTKEY   hDrvKey = 0;
  char    buf[4096];
  int     retval=1;

  char    *pszContainer = (pargs->lpszContainerName ? 
                                strdup(pargs->lpszContainerName) : NULL);

  printf(MODULE
        "!fa_selftest:%d: -container %s -device %s "
        "-numblocks %u\n", __LINE__,
        (pargs->lpszContainerName ? pargs->lpszContainerName : "<NULL>"),
        (pargs->lpszDeviceName ? pargs->lpszDeviceName : "<NULL>"),
        (unsigned)pargs->siNumBlocks);
// Numblocks is guaranteed to me small	
        
  if (!pargs->lpszDeviceName || pargs->siNumBlocks > 1000000){
        printf(MODULE "!fa_selftest:%d: Bad args\n", __LINE__);
        goto err_ret;
  }
    
  if (csp->AcquireContext(csp, &hProv, pszContainer, 0, CurVTable)) {
        printf(MODULE "!fa_selftest:%d: AcquireContext\n", __LINE__);
        goto err_ret;
  }
 
#ifdef USE_DAEMON
  int nest_retval;
  nest_retval=daemon_handshake(pargs, hProv, &drv, &hDrvKey);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */

  for (unsigned int i = 0; i < pargs->siNumBlocks; i++) {
#ifdef USE_DAEMON
  nest_retval=daemon_kernel_selftest(drv);
  if(nest_retval) {
      retval = nest_retval;
      goto err_ret;
  }
#endif /* USE_DAEMON */
  }

  memset(&buf, 0, sizeof(buf));

  printf(MODULE "!fa_selftest:OK\n");
  retval = 0;
    
err_ret:;

  if (hDrvKey) csp->DestroyKey(csp, hProv, hDrvKey);
  if (hProv) csp->ReleaseContext(csp, hProv, 0);
  free(pszContainer);
    
  memset(&buf, 0, sizeof(buf));

  if(drv!=-1) close(drv);

  return retval;
}
