#ifndef _IKEV2G_KTREE_H_
#define _IKEV2G_KTREE_H_

#include "blob.h"
#include "types.h"

typedef struct mgm_ktree_t mgm_ktree_t;

/** TC26 MGM + KTREE */
struct mgm_ktree_t {

    /**
     * @brief Encrypt inplace and sign data with associated data.
     *
     * The plain data must include padding and must be a multiple of some
     * number of bytes required by encryption algorithm.
     * MGM MAGMA and KUZNYECHIK do not require some special size of a plaintext.
     * Provided plain data must have additional space for ICV.
     * So plain.ptr buffer size must be at least plain.len + icv_size.
     *
     * @param plain	Blob to encrypt and sign, where result will be stored.
     * @param assoc	Associated data to sign.
     * @param iv	Initialization vector.
     * @return		True on success.
     */
    bool(*encrypt_data)(mgm_ktree_t *this, blob_t plain, blob_t assoc, blob_t iv)
	WARN_UNUSED_RESULT;

    /**
     * @brief Encrypt inplace and sign data with associated data.
     *
     * The plain data must include padding and must be a multiple of some
     * number of bytes required by encryption algorithm.
     * MGM MAGMA and KUZNYECHIK do not require some special size of a plaintext.
     * Provided plain data must have additional space for ICV.
     * So plain.ptr buffer size must be at least plain.len + icv_size.
     *
     * @param plain	IOV to encrypt and sign, where result will be stored.
     * @param assoc	Associated data to sign.
     * @param iv	Initialization vector.
     * @return		True on success.
     */
    bool(*encrypt_iov)(mgm_ktree_t *this, blob_t plain, blob_t assoc, blob_t iv)
	WARN_UNUSED_RESULT;

    /**
     * @brief Check if key tree is exhausted.
     *
     * Checks if encryption failed due to bad key tree status.
     * Key tree considered "depleted" if all available combinations
     * of i1, i2, i3 already been used.
     *
     * @return		True if key tree depleted.
     */
    bool(*ktree_exhausted)(mgm_ktree_t *this)
	WARN_UNUSED_RESULT;

    /**
     * @brief Decrypt inplace and check sign.
     *
     * The ecnrypted data must contain ICV tail. Decryption do not modify
     * ICV tail.
     *
     * @param plain	Blob to decrypt and check signature.
     * @param assoc	Associated data to check signature.
     * @param iv	Initialization vector.
     * @return		True on success.
     */
    bool(*decrypt_data)(mgm_ktree_t *this, blob_t encrypted, blob_t assoc, blob_t iv)
	WARN_UNUSED_RESULT;

    /**
     * @brief Decrypt inplace and check sign.
     *
     * The ecnrypted data must contain ICV tail. Decryption do not modify
     * ICV tail.
     *
     * @param plain	IOV to decrypt and check signature.
     * @param assoc	Associated data to check signature.
     * @param iv	Initialization vector.
     * @return		True on success.
     */
    bool(*decrypt_iov)(mgm_ktree_t *this, blob_t encrypted, blob_t assoc, blob_t iv)
	WARN_UNUSED_RESULT;

    /**
     * @brief Destructor of mgm_ktree_t.
     *
     * @param this	Pointer to the mgm_ktree_t structure.
     */
    void (*destroy)(mgm_ktree_t *this);

    /**
     * @brief Finalization of mgm_ktree_t without memory freeing.
     *
     * @param this		Pointer to the mgm_ktree_t structure.
     * @param release_prov	If CSP context should be release.
     */
    void (*fini)(mgm_ktree_t *this, bool release_prov );

};

/**
 * @brief Get size of the mgm_ktree_t object.
 *
 * @return	Size of the mgm_ktree_t object.
 */
size_t mgm_ktree_objsize(void);

/**
 * @brief Create MGM KTREE AEAD with specified algorithm.
 *
 * Create MT-unsafe version of mgm_ktree_t.
 *
 * @param prov		CSP context.
 * @param alg_id	Encryption algorithm id.
 * @param import_key	Key to import encryption or decryption key.
 * @param key_blob	Encrypted key to encrypt or decrypt data.
 * @param icv_size	Length of ICV.
 * @return		Pointer to the mgm_ktree_t structure on success.
 */
mgm_ktree_t *mgm_ktree_create(uintptr_t prov, int alg_id, uintptr_t import_key,
    blob_t key_blob, int icv_size);

/**
 * @brief Init allocated MGM KTREE AEAD with specified algorithm.
 *
 * Initialize MT-unsafe version of mgm_ktree_t.
 *
 * @param ktree		Pointer to the uninitialized mgm_ktree_t structure.
 * @param prov		CSP context.
 * @param alg_id	Encryption algorithm id.
 * @param import_key	Key to import encryption or decryption key.
 * @param key_blob	Encrypted key to encrypt or decrypt data.
 * @param icv_size	Length of ICV.
 * @return		True on success.
 */
bool mgm_ktree_init(mgm_ktree_t *ktree, uintptr_t prov, int alg_id,
    uintptr_t import_key, blob_t key_blob, int icv_size);

/**
 * @brief Create MGM KTREE AEAD with specified algorithm.
 *
 * Create MT version of mgm_ktree_t.
 *
 * @param alg_id	Encryption algorithm id.
 * @param keysalt_blob	Concatenation of a key and a salt.
 *			Key will be imported with import key from @p key_secret.
 * @param key_secret	Material to derive import key.
 * @param icv_size	Length of ICV.
 * @param coreid_fn	Service callback - get current core/thread id.
 * @param lock_fn	Service callback - mutex lock.
 * @param unlock_fn	Service callback - mutex unlock.
 * @return		Pointer to the mgm_ktree_t structure on success.
 */
mgm_ktree_t *mgm_ktree_create_mp(int alg_id, blob_t key_blob,
    blob_t key_secret, int icv_size,
    int (*coreid_fn)(void), void (*lock_fn)(void *),
    void (*unlock_fn)(void *));

/**
 * @brief Init MGM KTREE AEAD with specified algorithm.
 *
 * Initialize MT version of mgm_ktree_t.
 *
 * @param ktree		Pointer to the uninitialized mgm_ktree_t structure.
 * @param alg_id	Encryption algorithm id.
 * @param keysalt_blob	Concatenation of a key and a salt.
 *			Key will be imported with import key from @p key_secret.
 * @param key_secret	Material to derive import key.
 * @param icv_size	Length of ICV.
 * @param coreid_fn	Service callback - get current core/thread id.
 * @param lock_fn	Service callback - mutex lock.
 * @param unlock_fn	Service callback - mutex unlock.
 * @return		True on success.
 */
bool mgm_ktree_init_mp(mgm_ktree_t *ktree, int alg_id, blob_t key_blob,
    blob_t key_secret, int icv_size,
    int (*coreid_fn)(void), void (*lock_fn)(void *),
    void (*unlock_fn)(void *));

#endif /* _IKEV2G_KTREE_H_ */
