/* INCERT LICENCE HERE */

/**
 * @defgroup ikev2_gost ikev2_gost
 * @{ @ingroup ikev2
 */

#ifndef _IKEV2_GOST_H_
#define _IKEV2_GOST_H_

#ifdef USE_DETECTIVE
#include <stdio.h>
#endif

#include "types.h"
#include "blob.h"
#include "dh.h"
#include "ktree.h"
#include "prf.h"

typedef struct ikev2_gost_t ikev2_gost_t;

/**
 * Crypto-Pro IKEv2 interface.
 */
struct ikev2_gost_t {

    /*
     * @brief Derive IKE SA keys.
     *
     * For more information see 4.13.1 and 4.13.3 at MR 26.2.001-2022.
     *
     * @param this		Pointer to the ikev2_gost_t structure.
     * @param nonce_i		Initiator's nonce.
     * @param nonce_r		Responder's nonce.
     * @param spi_i		Initiator's SPI.
     * @param spi_r		Responder's SPI.
     * @param shared_key	Key derived during DH exchange.
     * @param aead_alg		Type of the ENCR transform for IKEv2 exchange.
     * @param rekey		Set to true if it's a rekey procedure.
     * @param rekey_skd		SKd key for rekeying if @p rekey is true.
     * @param export_key	Key to encrypt output @p key_i and @p key_r.
     * @param key_i		Encrypted key to encrypt data from a initiator.
     * @param key_r		Encrypted key to encrypt data from a responder.
     * @return			True on success.
     */
    bool (*derive_ike_keys)(ikev2_gost_t *this, blob_t nonce_i, blob_t nonce_r,
                            uint64_t spi_i, uint64_t spi_r,
			    uintptr_t shared_key, IKE_ALG_ID aead_alg,
			    bool rekey, uintptr_t rekey_skd,
                            uintptr_t export_key, blob_t *key_i, blob_t *key_r);

    /**
     * @brief Derive IPsec SA keys of an arbitrary size.
     *
     * For more information see 4.13.2 at MR 26.2.001-2022.
     *
     * @param this		Pointer to the ikev2_gost_t structure.
     * @param nonce_i		Initiator's nonce.
     * @param nonce_r		Responder's nonce.
     * @param key_size		Keys' length.
     * @param key_i		Key to encrypt data from a initiator.
     * @param key_r		Key to encrypt data from a responder.
     * @return			True on success.
     */
    bool(*derive_foreign_keys)(ikev2_gost_t *this,
				blob_t nonce_i, blob_t nonce_r,
				unsigned key_size,
				blob_t *key_i, blob_t *key_r);

    /**
    *@brief Derive IPsec SA keys.
    *
    * For more information see 4.13.2 at MR 26.2.001 - 2022.
    *
    * @param this		Pointer to the ikev2_gost_t structure.
    * @param nonce_i		Initiator's nonce.
    * @param nonce_r		Responder's nonce.
    * @param dh_secret		DH key for PFS mode.
    * @param aead_alg		Type of the ENCR transform for ESP.
    * @param export_key	Key to encrypt output @p key_i and @p key_r.
    * @param key_i		Encrypted key to encrypt data from a initiator.
    * @param key_r		Encrypted key to encrypt data from a responder.
    * @return			True on success.
    */
    bool(*derive_native_keys)(ikev2_gost_t *this,
				blob_t nonce_i, blob_t nonce_r,
				uintptr_t dh_secret, IKE_ALG_ID alg,
				uintptr_t export_key,
				blob_t *key_i, blob_t *key_r);

    /**
    * @brief Get auth blob.
    *
    * Generate authentication blob for itself or an opposite side.
    * Result blob must be freed after using.
    *
    * @param verify		Must be set to false to generate blob for itself
    *				and true for the opposite side.
    * @param ike_sa_init	Binary representaion of the last IKE_SA_INIT
    *				of the corresponding side.
    * @param nonce		Nonce of the opposite side. If we generate
    *				blob for the Initiator it must be nonce_r.
    * @param id		Identification section of the corresponding side
    *				excluding header (first 4 bytes).
    * @param octets		Uninitialized or empty blob where the result
    *				will be stored.
    * @return			True on success.
    */
    bool (*get_auth_octets)(ikev2_gost_t *this, bool verify, blob_t ike_sa_init,
	                    blob_t nonce, blob_t id, blob_t *octets);
    /**
    * @brief Get PSK auth signature.
    *
    * Building PSK auth signature for initiator or responder.
    *
    * @param this		Pointer to the ikev2_gost_t structure.
    * @param verify		If true then build auth octets of the opposite
    *				side.
    * @param secret		PSK.
    * @param ike_sa_init	Latest IKE_SA_INIT message (from ike header)
    *				sent by the side for which we are getting
    *				auth signature.
    * @param nonce		Nonce of the opposite side. If we generate
    *				signature for the Initiator it must be nonce_r.
    * @param id		Identification section of the corresponding side
    *				excluding header (first 4 bytes).
    * @param sig		Empty or uninitialized blob where signature
    *				will be stored.
    * @return			True on success.
    */
    bool (*get_psk_sig)(ikev2_gost_t *this, bool verify, blob_t secret,
                        blob_t ike_sa_init, blob_t nonce, blob_t id,
			blob_t *sig);

    /*
     * @brief Get SKd key.
     *
     * Get SKd key to use it for IKEv2 rekeing or creating Child SA.
     *
     * @param this	Pointer to the ikev2_gost_t structure.
     * @return		Key handle on success or 0 on failure.
     */
    uintptr_t (*get_skd)(ikev2_gost_t *this);

    /**
     * @brief Destructor of ikev2_gost_t.
     *
     * @param this	Pointer to the ikev2_gost_t structure.
     */
    void (*destroy)(ikev2_gost_t *this);
};

/**
 * @brief Constructor of ikev2_gost_t.
 *
 * @param initiator	Create IKEv2 interface for an Initiator or Responder.
 * @param prov		CSP context.
 * @return		Pointer to the created IKEv2 interface on success
 *			or NULL otherwise.
 */
ikev2_gost_t *ikev2_gost_create(bool initiator, uintptr_t prov);

/**
 * @brief Get (acquire and cache) CSP context of current thread.
 *
 * Usefull in case of multithread application.
 *
 * @return		CSP context (prov) on success or 0 otherwise.
 */
uintptr_t get_thread_context(void);

/*
 * @brief Acquire new CSP context.
 *
 * @return		Returns CSP context (prov) on success or 0 otherwise.
 */
uintptr_t ikev2g_acquire_context(void);

/**
 * @brief Release previously acquired CSP context.
 *
 * @param prov		CSP context to release.
 */
void ikev2g_release_context(uintptr_t prov);

/**
 * @brief Generate random bytes.
 *
 * @param buf		Buffer of length at least @p count, where random bytes
 *			will be stored.
 * @param count		Total number of random bytes.
 * @return		True on success.
 */
bool ikev2g_rand(uint8_t *buf, int count);

/**
 * @brief Derive key as hash of secret.
 *
 * Derive key that could be used as an export key.
 *
 * @param prov		CSP context.
 * @param secret	Random data to generate key.
 * @return		Key handler on success or 0 otherwise.
 */
uintptr_t ikev2g_derive_key(uintptr_t prov, blob_t secret);

/**
 * @brief Derive exportable key as hash of secret.
 *
 * @param prov		CSP context.
 * @param secret	Random data to generate key.
 * @return		Key handler on success or 0 otherwise.
 */
uintptr_t ikev2g_derive_key_ex(uintptr_t prov, blob_t secret);

/**
 * @brief Export key and encrypt.
 *
 * @param prov		CSP context.
 * @param key		Key to export.
 * @param export_key	Key to encrypt exported key.
 * @param blob		Ininitialized or empty blob where encrypted exported
 *			key will be stored.
 * @return		True on success.
 */
bool ikev2g_export_key(uintptr_t prov, uintptr_t key, uintptr_t export_key,
			blob_t *blob);

/**
 * @brief Import encrypted key.
 *
 * Decrypt and import exported key.
 *
 * @param prov		CSP context.
 * @param import_key	Key to decrypt.
 * @param blob		Blob with encrypted exported key.
 * @return		Imported key handler on success or 0 otherwise.
 */
uintptr_t ikev2g_import_key(uintptr_t prov, uintptr_t import_key, blob_t blob);

/**
 * @brief Clear key handler.
 *
 * @param key		Key to destroy.
 */
void ikev2g_destroy_key(uintptr_t key);

#ifdef USE_DETECTIVE
/**
 * @brief Handle-leak detective report.
 *
 * Print to the filestream number of alive crypto objects and addresses
 * of their handlers.
 *
 * @param out		File descriptor where detective report will be stored.
 */
void ikev2g_detective_report(FILE *out);
#endif /* USE_DETECTIVE */

/**
 * @brief Load license vendor id.
 *
 * Load PP_RESERVED5 from CSP handler.
 *
 * @param buf		Buffer where license vendor id will be stored.
 * @param plen		Length of license vendor id.
 * @return		True on success.
 */
bool ikev2g_load_lic_vendor_id(char* buf, uint32_t *plen);

#ifdef IPSEC_CONFORMITY
/* Conformity test functions */
bool ipsec_conformity_is_enabled(void);
void ipsec_conformity_get_ike_spi(unsigned char **ppBuf, size_t *plen);
void ipsec_conformity_get_ephemeral(unsigned char **ppBuf, size_t *plen);
void ipsec_conformity_get_nonce(unsigned char **ppBuf, size_t *plen);
void ipsec_conformity_get_vendor_id(unsigned char **ppBuf, size_t *plen);
void ipsec_conformity_get_src_ip_port(unsigned char *ip_addr, uint16_t *port);
void ipsec_conformity_get_dst_ip_port(unsigned char *ip_addr, uint16_t *port);
void ipsec_conformity_get_esp_spi(unsigned char **ppBuf, size_t *plen);
void ipsec_conformity_get_ike_auth_plain(unsigned char **ppBuf, size_t *plen);
void ipsec_conformity_get_ike_auth_encr(unsigned char **ppBuf, size_t *plen);
#endif /* IPSEC_CONFORMITY */

#endif /** _IKEV2_GOST_H_ @} */
