#ifndef _IKEV2G_DH_H_
#define _IKEV2G_DH_H_

#include "blob.h"
#include "types.h"

typedef struct gost_dh_t gost_dh_t;

/**
 * @brief Class to handle DH exchange.
 */

struct gost_dh_t
{
    /**
	* @brief Get local DH public key.
	*
	* Get own (local) public key for the DH exchange.
	*
	* @param this	Pointer to the gost_dh_t structure.
	* @param value	Empty or uninitialized blob where public key will be
	*			stored. Memory freeing for this is required.
	* @return		Returns true on success.
	*/
    bool (*get_my_public)(gost_dh_t *this, blob_t *value);

    /**
	* @brief Set remote DH public key.
	*
	* Set other (remote) public key from DH exchange and generate shared key.
	*
	* @param this	Pointer to the gost_dh_t structure.
	* @param value	Blob with opposite's side public key received during
	*			DH exchange.
	* @return		Returns true on success.
	*/
    bool (*set_other_public)(gost_dh_t *this, blob_t value);

    /**
	* @brief Get DH shared key handle.
	*
	* @param this	Pointer to the gost_dh_t structure.
	* @return		Pointer to the shared key handle or 0 on failure.
	*/
    uintptr_t (*get_shared_key)(gost_dh_t *this);


    /**
	* @brief Destroy DH object.
	*
	* @param this	Pointer to the gost_dh_t structure.
	*/
    void (*destroy)(gost_dh_t *this);
};


/**
 * @brief Create GOST (EC)DH object.
 *
 * Generate random assymetric key and create object to compute DH keys.
 * Additionaly exports KP_RESERVED1 (X) parameter of the key if supported.
 *
 * @param prov		CSP handle.
 * @param group		DH group ID.
 * @param nonce_ptr	Allocated space for KP_RESERVED1 if supported.
 * @param nonce_len	Pointer to the nonce_len length.
 * @return		Pointer to the allocated DH object.
 */
gost_dh_t *gost_dh_create(uintptr_t prov, IKE_DH_GROUP, char *nonce_ptr, int *nonce_len);

#endif /* _IKEV2G_DH_H_ */
