/*
 * Copyright (C) 1997-2003 by Objective Systems, Inc.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by Objective Systems, Inc.
 *
 * PROPRIETARY NOTICE
 *
 * This software is an unpublished work subject to a confidentiality agreement
 * and is protected by copyright and trade secret law.  Unauthorized copying,
 * redistribution or other use of this work is prohibited.
 *
 * The above notice of copyright on this source code product does not indicate
 * any actual or intended publication of such source code.
 *
 *****************************************************************************/
/*
// CHANGE LOG
// Date         Init    Description
// 11/03/03	CZ	Expand documentation comments.
//
//////////////////////////////////////////////////////////////////////////////
*/

/** 
 * @file asn1xer.h 
 * ASN.1 runtime constants, data structure definitions, and functions to
 * support the XML Encoding Rules (XER) as defined in the ITU-T X.693 standard.
 */

/**
 * @mainpage ASN1C XER Runtime Classes and Library Functions
 *
 * The <B>ASN.1 C++ runtime classes</B> are wrapper classes that provide
 * an object-oriented interface to the ASN.1 C Runtime Library functions.
 * The classes described in this manual are derived from the common 
 * classes documented in the ASN1C C/C++ Common Runtime manual. 
 * They are specific to the XML Encoding Rules (XER)as defined in
 * the X.693 ITU-T  standard. 
 * 
 * These XER specific C++ runtime classes include:
 * <UL> <LI>classes for streaming XER decoding </LI><LI>classes for 
 * streaming XER encoding.</LI></UL>
 *
 * The <B> ASN.1 XER Runtime Library</B> contains the low-level 
 * constants, types, and functions that are assembled by the compiler 
 * to encode/decode more complex structures.
 *
 * This library consists of the following items:
 *   <UL> <LI>A global include file ("asn1xer.h") that is compiled into all
 * generated source files.</LI>
 *<LI> An object library of functions that are linked with the C functions
 * after compilation with a C complier.</LI></UL>
 *
 * In general, programmers will not need to be too concerned with the details
 * of these functions. The ASN.1 compiler generates calls to them in the C or
 * C++ source files that it creates. However, the functions in the library may
 * also be called on their own in applications requiring their specific
 * functionality.
 */
#ifndef _ASN1XER_H_
#define _ASN1XER_H_

#if !defined NO_ASN1_RT_REDUNDANT
#   define NO_ASN1_RT_REDUNDANT
#endif /* !NO_ASN1_RT_REDUNDANT */

#include "rtsrc/asn1type.h"
#include "rtSAXDefs.h"

/** @defgroup xerruntime XER Runtime Library Functions.
 * @{
 * The ASN.1 XML Encoding Rules (XER) runtime library contains the  
 * low-level constants, types, and functions that are assembled by the compiler 
 * to encode/decode more complex structures. 
 *
 * The XER low-level C encode/decode functions are
 * identified by their prefixes: xerEnc for XER encode, 
 * xerDec for XER decode, and xer for XER utility functions. 
 * 
 */

typedef enum { 
   XERINIT, 
   XERSTART, 
   XERDATA, 
   XEREND, 

   XERSTART0, 
   XEREND0
} ASN1XERState;

#define XERINDENT       3

typedef struct {
   ASN1ConstCharPtr name;
   ASN1BOOL optional;
} XerElemInfo;

typedef struct XmlNamedBitsDict {
   ASN1ConstCharPtr name;
   ASN1UINT bitnum;
} XmlNamedBitsDict;

/* This macro returns the byte count in the given context buffer */
#define XERBYTECNT(pctxt) (pctxt)->buffer.byteIndex

#ifdef __cplusplus
extern "C" {
#endif

#ifdef BUILDASN1XERDLL
#define EXTERNXER __declspec(dllexport)
#elif defined (USEASN1XERDLL)
#define EXTERNXER __declspec(dllimport)
#else
#define EXTERNXER
#endif /* BUILDASN1XERDLL */

#ifdef USEASN1XERDLL
#define DLLNOINLINEXER __declspec(noinline)
#else
#define DLLNOINLINEXER
#endif // USEASN1XERDLL

/* decode functions */
/** @defgroup xerdecruntime XER C Decode Functions.
 * @{
 *
 * XER runtime library decode functions handle the decoding of the primitive
 * ASN.1 data types and length variables. Calls to these functions are
 * assembled in the C source code generated by the ASN1C complier to decode
 * complex ASN.1 structures. These functions are also directly callable from
 * within a user's application program if the need to decode a primitive data
 * item exists.
 */

/** 
 * This function will decode a variable ASN.1 16-bit character BMPString type.
 * This function will allocate dynamic memory to store the decoded result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param outdata      A pointer to a structure variable to receive the decoded
 *                       string. The string is stored as an array of short
 *                       integer characters. The memory is allocated for the
 *                       string by the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecBMPStr (ASN1CTXT* pctxt, 
                            ASN1BMPString* outdata);

/**
 * This function will decode a variable of teh ASN.1 OCTET STRING type into a
 * static memory structure. The octet string must be Base64 encoded. This
 * function call is used to decode a sized octet string production.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a variable to receive the decoded bit
 *                       string. This is assumed to be a static array large
 *                       enough to hold the number of octets specified in the
 *                       bufsize input parameter.
 * @param pnocts       A pointer to an integer value to receive the decoded
 *                       number of octets.
 * @param bufsize      A integer variable containing the size (in octets) of
 *                       the sized ASN.1 octet string. An error will occur if
 *                       the number of octets in the decoded string is larger
 *                       than this value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecBase64Str (ASN1CTXT* pctxt, 
                               ASN1OCTET* pvalue, ASN1UINT* pnocts, 
                               ASN1INT bufsize);

/**
 * This function will decode a variable of the ASN.1 INTEGER type. In this
 * case, the integer is assumed to be of a larger size than can fit in a C or
 * C++ long type (normally 32 or 64 bits). For example, parameters used to
 * calculate security values are typically larger than these sizes.
 *
 * These variables are stored in character string constant variables. Depending
 * on specified radix, they could be represented as binary, octal, decimal or
 * hexadecimal strings starting with appropriate prefix. If it is necessary to
 * convert to another radix, then use rtBigIntSetStr or rtBigIntToString
 * functions.
 *
 * @param pctxt        Pointer to context block structure.
 * @param ppvalue      Pointer to a character pointer variable to receive the
 *                       decoded unsigned value. Dynamic memory is allocated
 *                       for the variable using the rtMemAlloc function. The
 *                       decoded variable is represented as a string starting
 *                       with appropriate prefix.
 * @param radix        The expected radix of the decoded value. The only
 *                       radices 2, 8, 10 and 16 are supported.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecBigInt (ASN1CTXT* pctxt, char **ppvalue, int radix);


/**
 * This function will decode a variable of the ASN.1 BIT STRING type into a
 * static memory structure. This function call is generated by ASN1C to decode
 * a sized bit production.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a variable to receive the decoded bit
 *                       string. This is assumed to be a static array large
 *                       enough to hold the number of octets specified in the
 *                       bufsize input parameter.
 * @param pnbits       A pointer to an integer value to receive the decoded
 *                       number of bits.
 * @param bufsize      An integer variable containing the size (in octets) of
 *                       the sized ASN.1 bit string. An error will occur if the
 *                       number of octets in the decoded string is larger than
 *                       this value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecBitStr (ASN1CTXT* pctxt, ASN1OCTET* pvalue, 
                            ASN1UINT* pnbits, ASN1INT bufsize);

/**
 * This function decodes a variable of the ASN.1 BIT STRING type to a memory
 * buffer. The decoded data will be put into the memory buffer starting from
 * the current position and bit offset. After all data is decoded the bit
 * string may be fetched out by call to ::xerDecCopyBitStr or
 * ::xerDecCopyDynBitStr functions. <p>Usually, this function is used in the
 * 'characters' SAX handler.
 *
 * @param pMemBuf            Pointer to the destination memory buffer.
 * @param inpdata            Pointer to a source string to be decoded.
 * @param length             Length of the source string (in characters).
 * @param skipWhitespaces    Indicates, could whitespaces be ignored or they
 *                             are illegal.
 * @return                   Completion status of operation:
 *                             - 0 (ASN_OK) = success,
 *                             - negative return value is error.
 */
EXTERNXER int xerDecBitStrMemBuf (ASN1MemBuf* pMemBuf, 
                                  ASN1Const XMLCHAR* inpdata, 
                                  int length, ASN1BOOL skipWhitespaces);

/**
 * This function decodes a variable of the ASN.1 BOOLEAN type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a variable to receive the decoded BOOLEAN
 *                       value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecBool (ASN1CTXT* pctxt, ASN1BOOL *pvalue);

/**
 * This function copies the decoded BIT STRING from the memory buffer. This
 * function call is generated by ASN1C to decode a sized bit string production.
 *
 * @param pctxt            Pointer to context block structure. Its buffer
 *                           should be set to point to the decoded data.
 * @param pvalue           Pointer to a buffer to receive the decoded data.
 * @param pnbits           Pointer to an integer value to receive the decoded
 *                           number of bits.
 * @param bufsize          An integer variable containing the size (in octets)
 *                           of the sized ASN.1 bit string. An error will occur
 *                           if the number of octets in the decoded string is
 *                           larger than this value.
 * @param lastBitOffset    A number of actual bits in the last octet.
 * @return                 Completion status of operation:
 *                           - 0 (ASN_OK) = success,
 *                           - negative return value is error.
 */
EXTERNXER int xerDecCopyBitStr (ASN1CTXT* pctxt, 
                                ASN1OCTET* pvalue, ASN1UINT* pnbits, 
                                ASN1INT bufsize, int lastBitOffset);

/**
 * This function copies the decoded BIT STRING from the memory buffer. This
 * function will allocate dynamic memory to store the decoded result.
 *
 * @param pctxt            Pointer to context block structure. Its buffer
 *                           should be set to point to the decoded data.
 * @param pvalue           Pointer to a dynamic bit string structure to receive
 *                           the decoded bit string. Dynamic memory is
 *                           allocated to hold the string using the
 *                           ::rtMemAlloc function.
 * @param lastBitOffset    A number of actual bits in the last octet.
 * @return                 Completion status of operation:
 *                           - 0 (ASN_OK) = success,
 *                           - negative return value is error.
 */
EXTERNXER int xerDecCopyDynBitStr (ASN1CTXT* pctxt, 
                                   ASN1DynBitStr* pvalue, 
                                   int lastBitOffset);

/**
 * This function copies the decoded OCTET STRING from the memory buffer. This
 * function will allocate dynamic memory to store the decoded result.
 *
 * @param pctxt            Pointer to context block structure. Its buffer
 *                           should be set to point to the decoded data.
 * @param pvalue           Pointer to a dynamic octet string structure to
 *                           receive the decoded octet string. Dynamic memory
 *                           is allocated to hold the string using the
 *                           ::rtMemAlloc function.
 * @param lastBitOffset    A number of actual bits in the last octet.
 * @return                 Completion status of operation:
 *                           - 0 (ASN_OK) = success,
 *                           - negative return value is error.
 */
EXTERNXER int xerDecCopyDynOctStr (ASN1CTXT* pctxt, ASN1DynOctStr* pvalue, 
                                   int lastBitOffset);

/**
 * This function copies the decoded OCTET STRING from the memory buffer. This
 * function call is generated by ASN1C to decode a sized octet string
 * production.
 *
 * @param pctxt            Pointer to context block structure. Its buffer
 *                           should be set to point to the decoded data.
 * @param pvalue           Pointer to a variable to receive the decoded bit
 *                           string. This is assumed to be a static array large
 *                           enough to hold the number of octets specified in
 *                           the 'bufsize' input parameter.
 * @param pnocts           Pointer to an integer value to receive the decoded
 *                           number of octets.
 * @param bufsize          An integer variable containing the size (in octets)
 *                           of the sized ASN.1 octet string. An error will
 *                           occur if the number of octets in the decoded
 *                           string is larger than this value.
 * @param lastBitOffset    A number of actual bits in the last octet.
 * @return                 Completion status of operation:
 *                           - 0 (ASN_OK) = success,
 *                           - negative return value is error.
 */
EXTERNXER int xerDecCopyOctStr (ASN1CTXT* pctxt, 
                                ASN1OCTET* pvalue, ASN1UINT* pnocts, 
                                ASN1INT bufsize, int lastBitOffset);

/** 
 * This function will decode a variable of one of the ASN.1 8-bit character
 * string types. These types include IA5String, VisibleString, PrintableString,
 * and NumericString. This function will allocate dynamic memory to store the
 * result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param outdata      A pointer to a character string pointer variable to
 *                       receive the decoded string. The string as stored as a
 *                       standard null-terminated C string. Memory is allocated
 *                       for the string by the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecDynAscCharStr (ASN1CTXT* pctxt, 
                                   ASN1ConstCharPtr* outdata);


/**
 * This function will decode a variable of the ASN.1 OCTET STRING type. The
 * octet string must be Base64 encoded. This function will allocate dynamic
 * memory to store the decoded result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a dynamic octet string structure to receive
 *                       the decoded octet string. Dynamic memory is allocated
 *                       to hold the string using the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecDynBase64Str (ASN1CTXT* pctxt, ASN1DynOctStr* pvalue);

/**
 * This function will decode a variable of the ASN.1 BIT STRING type. This
 * function will allocate dynamic memory to store the decoded result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a dynamic bit string structure to receive
 *                       the decoded bit string. Dynamic memory is allocated to
 *                       hold the string using the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecDynBitStr (ASN1CTXT* pctxt, ASN1DynBitStr* pvalue);

/** 
 * This function will decode a variable of the ASN.1 OCTET STRING type. This
 * function will allocate dynamic memory to store the decoded result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a dynamic bit string structure to receive
 *                       the decoded bit string. Dynamic memory is allocated to
 *                       hold the string using the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecDynOctStr (ASN1CTXT* pctxt, ASN1DynOctStr* pvalue);

/**
 * This function will decode a variable of UTF8String ASN.1 type. Generally,
 * the SAX parser converts all UTF8 format strings to 16-bit Unicode format
 * automatically. This function converts the Unicode string back to UTF8
 * format. This function will allocate dynamic memory to store the decoded
 * result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param outdata      A pointer to a character string pointer variable to
 *                       receive the decoded string. The string as stored as a
 *                       UTF8 null-terminated string. Memory is allocated for
 *                       the string by the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecDynUTF8Str (ASN1CTXT* pctxt, 
                                ASN1UTF8String* outdata);

/**
 * This function decodes a variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a variable to receive the decoded integer
 *                       value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecInt (ASN1CTXT* pctxt, ASN1INT* pvalue);

/**
 * This function decodes an 8-bit variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to an 8-bit variable to receive the decoded
 *                       integer value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNXER int xerDecInt8 (ASN1CTXT* pctxt, ASN1INT8* pvalue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function decodes a 16-bit variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a 16-bit variable to receive the decoded
 *                       integer value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNXER int xerDecInt16 (ASN1CTXT* pctxt, ASN1SINT* pvalue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function decodes a 64-bit variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a 64-bit variable to receive the decoded
 *                       integer value. The ASN1INT64 type is set to the C type
 *                       '__int64', 'long long' or 'long' in the asn1type.h
 *                       file (depends on the used platform and the compiler).
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecInt64 (ASN1CTXT* pctxt, ASN1INT64* pvalue);

/**
 * This function decodes a value of the ASN.1 OBJECT IDENTIFIER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to value to receive decoded result. The
 *                       ASN1OBJID structure contains an integer to hold the
 *                       number of subidentifiers and an array to hold the
 *                       subidentifier values.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecObjId (ASN1CTXT* pctxt, ASN1OBJID *pvalue);

/**
 * This function decodes a value of the ASN.1 OBJECT IDENTIFIER type using
 * 64-bit subidentifiers.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to value to receive decoded result. The
 *                       ASN1OID64 structure contains an integer to hold the
 *                       number of subidentifiers and an array of 64-bit
 *                       unsigned integers to hold the subidentifier values.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecObjId64 (ASN1CTXT* pctxt, ASN1OID64 *pvalue);

/**
 * This function will decode a variable of the ASN.1 OCTET STRING type into a
 * static memory structure. This function call is generated by ASN1C to decode
 * a sized octet string production.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a variable to receive the decoded bit
 *                       string. This is assumed to be a static array large
 *                       enough to hold the number of octets specified in the
 *                       bufsize input parameter.
 * @param pnocts       Pointer to an integer value to receive the number of
 *                       octets.
 * @param bufsize      An integer variable containing the size (in octets) of
 *                       the sized ASN.1 octet string. An error will occur if
 *                       the number of octets in the decoded string is larger
 *                       than this value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecOctStr (ASN1CTXT* pctxt, ASN1OCTET* pvalue, 
                            ASN1UINT* pnocts, ASN1INT bufsize);

/**
 * This function decodes a variable of the ASN.1 OCTET STRING type to a memory
 * buffer. The decoded data will be put into the memory buffer starting from
 * the current position and bit offset. After all data is decoded the octet
 * string may be fetched out by call to ::xerDecCopyOctStr or
 * ::xerDecCopyDynOctStr functions. <p>Usually, this function is used in the
 * 'characters' SAX handler.
 *
 * @param pMemBuf            Pointer to the destination memory buffer.
 * @param inpdata            Pointer to a source string to be decoded.
 * @param length             Length of the source string (in characters).
 * @param skipWhitespaces    Indicates, could whitespaces be ignored or they
 *                             are illegal.
 * @return                   Completion status of operation:
 *                             - 0 (ASN_OK) = success,
 *                             - negative return value is error.
 */
EXTERNXER int xerDecOctStrMemBuf (ASN1MemBuf* pMemBuf, 
                                  ASN1Const XMLCHAR* inpdata, 
                                  int length, ASN1BOOL skipWhitespaces);

/** 
 * This function will decode an ASN.1 open type. This used to be the ASN.1 ANY
 * type, but now is used in a variety of applications requiring an encoding
 * that can be interpreted by a decoder without prior knowledge of the type of
 * the variable.
 *
 * @param pctxt        A pointer to a context block structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a structure variable to receive the decoded
 *                       string. The string as stored as an array of unsigned
 *                       integer characters. Memory is allocated for the string
 *                       by the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecOpenType (ASN1CTXT* pctxt, ASN1OpenType* pvalue);

/** 
 * This function will decode a variable of the ASN.1 32-bit character
 * UniversalString type. This includes the UniversalString type.
 *
 * @param pctxt        A pointer to a context block structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a structure variable to receive the decoded
 *                       string. The string as stored as an array of unsigned
 *                       integer characters. Memory is allocated for the string
 *                       by the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecReal (ASN1CTXT* pctxt, ASN1REAL* pvalue);

/**
 * This function decodes a value of the ASN.1 RELATIVE-OID type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to value to receive decoded result. The
 *                       ASN1OBJID structure contains an integer to hold the
 *                       number of subidentifiers and an array to hold the
 *                       subidentifier values.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecRelativeOID (ASN1CTXT* pctxt, ASN1OBJID *pvalue);

/**
 * This function decodes a variable of the unsigned variant of ASN.1 INTEGER
 * type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a variable to receive the decoded unsigned
 *                       integer value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecUInt (ASN1CTXT* pctxt, ASN1UINT* pvalue);

/**
 * This function decodes an 8-bit variable of the unsigned variant of ASN.1
 * INTEGER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to an 8-bit variable to receive the decoded
 *                       unsigned integer value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNXER int xerDecUInt8 (ASN1CTXT* pctxt, ASN1UINT8* pvalue);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function decodes a 16-bit variable of the unsigned variant of ASN.1
 * INTEGER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a 16-bit variable to receive the decoded
 *                       unsigned integer value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecUInt16 (ASN1CTXT* pctxt, ASN1USINT* pvalue);

/**
 * This function decodes a 64-bit variable of the unsigned variant of ASN.1
 * INTEGER type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a 64-bit variable to receive the decoded
 *                       unsigned integer value. The ASN1UINT64 type is set to
 *                       the C type 'unsigned __int64', 'unsigned long long' or
 *                       'unsigned long' in the asn1type.h file (depends on the
 *                       used platform and the compiler).
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerDecUInt64 (ASN1CTXT* pctxt, ASN1UINT64* pvalue);

/** 
 * This function will decode a variable an ASN.1 32-bit character
 * UniversalString type. This includes the UniversalString type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param outdata      A pointer to a structure variable to receive the decoded
 *                       string. The string as stored as an array of unsigned
 *                       integer characters. Memory is allocated for the string
 *                       by the rtMemAlloc function.
 */
EXTERNXER int xerDecUnivStr (ASN1CTXT* pctxt, 
                             ASN1UniversalString* outdata);

/** 
 * This function is used to set the internal decode buffer pointer within the
 * runtime library decode module. It must be called prior to calling any other
 * complier generated or runtime library decode functions.
 *
 * @param pCtxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param bufaddr      A pointer to a memory buffer containing the ASN.1
 *                       message. The pointer must point at the first byte in
 *                       the message.
 * @param bufsiz       The size of the message that was read. This is used to
 *                       set an internal message size to check for field length
 *                       errors. If this size is not known, a zero value can be
 *                       passed to cause these checks to be bypassed.
 */
EXTERNXER int xerSetDecBufPtr (ASN1CTXT* pCtxt, ASN1ConstOctetPtr bufaddr, 
                               ASN1UINT bufsiz);

/** @} xerdecruntime  */

/* encode functions */
/** @defgroup xerencruntime XER C Encode Functions.
 * @{
 *
 * The XER low-level encode functions handle the XER encoding of primitive
 * ASN.1 data types. Calls to these functions are assembled in the C source
 * code generated by the ASN1C complier to accomplish the encoding of complex
 * ASN.1 structures. These functions are also directly callable from within a
 * user's application program if the need to accomplish a low level encoding
 * function exists.
 *
 * The procedure to call a low-level encode function is the same as the
 * procedure to call a complier generated encode function described above. The
 * rtInitContext and xerSetEncBufPtr functions must first be called to
 * initialize a context and set a pointer to a buffer into which the variable
 * is to be encoded. A static encode buffer is spcified by specifying a pointer
 * to a buffer and buffer size. Setting the buffer address to NULL and buffer
 * size to 0 specifies a dynamic buffer. The encode function is then invoked.
 * The result of the encoding will start at the beginning of the specified
 * buffer, or, if a dynamic buffer was used, it can be obtained by calling
 * xerGetMsgPtr. The length of the encoded component is obtained by calling
 * xerGetMsgLen.
 */

/**
 * This function is used to set the internal buffer within the runtime library
 * encode module. It must be called prior to calling any other generated or
 * runtime library encode functions.
 *
 * @param pCtxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param bufaddr      A pointer to a memory buffer containing the ASN.1
 *                       message. The pointer must point at the first byte in
 *                       the message.
 * @param bufsiz       The size of the message that was read. This is used to
 *                       set an internal message size to check for field length
 *                       errors. If this size is not known, a zero value can be
 *                       passed to cause these checks to be bypassed.
 * @param canonical    TRUE, if canonical XML encoding rules (CXER) should be
 *                       used. Otherwise FALSE.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerSetEncBufPtr (ASN1CTXT* pCtxt, ASN1OCTET* bufaddr, 
                               ASN1UINT bufsiz, ASN1BOOL canonical);

/** 
 * This function encodes a variable one of the ASN.1 character string types
 * that are based on 8-bit character sets. This includes IA5Sring,
 * VisibleString, PrintableString, and UTF8String, and NumericString.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a null-terminated C character string to be
 *                       encoded.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If an empty string
 *                       is passed (""), no element tag is added to the encoded
 *                       value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncAscCharStr (ASN1CTXT* pctxt, 
                                ASN1ConstCharPtr value, 
                                ASN1ConstCharPtr elemName);


/** 
 * This function encodes a variable of the ASN.1 OCTET STRING type using Base64
 * encoding.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param nocts        The number of octets (bytes) within the OCTET STRING to
 *                       be encoded.
 * @param data         A pointer to an OCTET STRING containing the octet data
 *                       to be encoded.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type is (<OCTET_STRING>) is
 *                       added. If an empty string is passed (""), no element
 *                       tag is added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncBase64Str (ASN1CTXT* pctxt, ASN1UINT nocts, 
                               ASN1ConstOctetPtr data, 
                               ASN1ConstCharPtr elemName);

/**
 * This function encodes a variable of the ASN.1 INTEGER type. In this case,
 * the integer is assumed to be of a larger size than can fit in a C or C++
 * long type (normally 32 or 64 bits). For example, parameters used to
 * calculate security values are typically larger than these sizes.
 *
 * Items of this type are stored in character string constant variables. They
 * can be represented as decimal strings (with no prefixes), as hexadecimal
 * strings starting with a "0x" prefix, as octal strings starting with a "0o"
 * prefix or as binary strings starting with a "0b" prefix. Other radixes are
 * currently not supported.
 *
 * @param pctxt        Pointer to context block structure.
 * @param value        A pointer to a character string containing the value to
 *                       be encoded.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<INTEGER\>) is added.
 *                       If an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncBigInt (ASN1CTXT* pctxt, ASN1ConstCharPtr value, 
                            ASN1ConstCharPtr elemName);

/** 
 * This function encodes a variable of the ASN.1 BIT STRING type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param nbits        The number of bits within the bit string to be encoded.
 * @param data         A pointer to an OCTET string containing the bit data to
 *                       be encoded. This string contains bytes having the
 *                       actual bit settings as they are to be encoded in the
 *                       message.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (<BIT_STRING>) is
 *                       added. If an empty string is passed (""), no element
 *                       tag is added to the encoded value.
 * @param outputType   An enumerated type whose value is set to either
 *                       'ASN1BIN' (for binary format) or 'ASN1HEX' (for
 *                       hexadecimal format).
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncBitStr (ASN1CTXT* pctxt, ASN1UINT nbits, 
                            ASN1ConstOctetPtr data, 
                            ASN1ConstCharPtr elemName, 
                            ASN1StrType outputType);

#if !defined NO_ASN1_RT_REDUNDANT
EXTERNXER int xerEncBoolValue (ASN1CTXT* pctxt, ASN1BOOL value);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/** 
 * This function encodes a variable of the ASN.1 BOOLEAN type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A BOOLEAN value to be encoded. A BOOLEAN is defined as a
 *                       single octet whose value is 0 for False and any other
 *                       value for TRUE.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (<BOOLEAN>) is added.
 *                       If an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncBool (ASN1CTXT* pctxt, ASN1BOOL value, 
                          ASN1ConstCharPtr elemName);

/** 
 * This function should be called at the end of the document's encoding.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncEndDocument (ASN1CTXT* pctxt);

/** 
 * This function encodes the ending tag of the XML element, such as </element>.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param elemName     A pointer to the name of the element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncEndElement (ASN1CTXT* pctxt, ASN1ConstCharPtr elemName);

/**
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 */
EXTERNXER int xerEncIndent (ASN1CTXT* pctxt);

/**
 * This function encodes a variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        An INTEGER value to be encoded. The ASN1INT type is set
 *                       to the C type 'int' in the asn1type.h file. This is
 *                       assumed to represent a 32-bit integer value.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<INTEGER\>) is added.
 *                       If an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncInt (ASN1CTXT* pctxt, ASN1INT value, 
                         ASN1ConstCharPtr elemName);

/**
 * This function encodes a 64-bit variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A 64-bit INTEGER value to be encoded. The ASN1INT64 type
 *                       is set to the C type '__int64', 'long long' or 'long'
 *                       in the asn1type.h file (depends on the used platform
 *                       and the compiler). This is assumed to represent a
 *                       64-bit integer value.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<INTEGER\>) is added.
 *                       If an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncInt64 (ASN1CTXT* pctxt, ASN1INT64 value, 
                           ASN1ConstCharPtr elemName);


/** 
 * This function inserts a new line symbol int the XML document.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncNewLine (ASN1CTXT* pctxt);

/**
 * This function encodes a variable of the ASN.1 OBJECT IDENTIFIER type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to an object identifier structure. This
 *                       structure contains an integer to hold the number of
 *                       subidentifers in the object and an array to hold the
 *                       subidentifier values.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<OBJECT_IDENTIFIER\>)
 *                       is added. If an empty string is passed (""), no
 *                       element tag is added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncObjId (ASN1CTXT* pctxt, const ASN1OBJID *pvalue, 
                           ASN1ConstCharPtr elemName);

/**
 * This function encodes a variable of the ASN.1 OBJECT IDENTIFIER type using
 * 64-bit subidentifiers.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a 64-bit object identifier structure. This
 *                       structure contains an integer to hold the number of
 *                       subidentifers in the object and an array of 64-bit
 *                       unsigned integers to hold the subidentifier values.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<OBJECT_IDENTIFIER\>)
 *                       is added. If an empty string is passed (""), no
 *                       element tag is added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncObjId64 (ASN1CTXT* pctxt, const ASN1OID64 *pvalue, 
                             ASN1ConstCharPtr elemName);

/**
 * This function encodes a variable of the ASN.1 RELATIVE-OID type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to an object identifier structure. This
 *                       structure contains an integer to hold the number of
 *                       subidentifers in the object and an array to hold the
 *                       subidentifier values.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<RELATIVE_OID\>) is
 *                       added. If an empty string is passed (""), no element
 *                       tag is added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncRelativeOID (ASN1CTXT* pctxt, const ASN1OBJID *pvalue, 
                                 ASN1ConstCharPtr elemName);

/** 
 * This function encodes a variable of the ASN.1 OCTET STRING type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param nocts        The number of octets (bytes) within the OCTET STRING to
 *                       be encoded.
 * @param data         A pointer to an OCTET STRING containing the octet data
 *                       to be encoded.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (<OCTET_STRING>) is
 *                       added. If an empty string is passed (""), no element
 *                       tag is added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncOctStr (ASN1CTXT* pctxt, ASN1UINT nocts, 
                            ASN1ConstOctetPtr data, 
                            ASN1ConstCharPtr elemName);

/** 
 * This function encodes a variable of the REAL data type. This function
 * provides support for the plus-infinity and the minus-infinity special real
 * values. Use the rtGetPlusInfinity or rtGetMinusInfinity functions to get
 * these special values.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A value to be encoded. Special real values plus and
 *                       minus infinity are encoded by using the
 *                       rtGetPlusInfinity and rtGetMinusInfinity functions to
 *                       set the real value to be encoded.
 * @param elemName     A pointer to the name of the element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncReal (ASN1CTXT* pctxt, ASN1REAL value, 
                          ASN1ConstCharPtr elemName);

/**  
 * This function encodes the starting record of the XML document (such as the
 * <?xml version = "1.0" encoding = "UTF-8"?>). This function should be called
 * prior to encoding any other fields in the document. After all elements in
 * the document are encoded, xerEncEndDocument should be called.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncStartDocument (ASN1CTXT* pctxt);

/**
 * This function encodes the string tag of the XML element, such as <element>.
 * After the element's data is encoded, the xerEncEndElement function should be
 * called.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param elemName     A pointer to the name of the element.
 * @param attributes   A pointer to the attributes of the element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncStartElement (ASN1CTXT* pctxt, 
                                  ASN1ConstCharPtr elemName, 
                                  ASN1ConstCharPtr attributes);

/** 
 * This function encodes an empty element, such as <element/>.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param elemName     A pointer to the name of the element.
 * @param attributes   A pointer to the attributes of the element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncEmptyElement (ASN1CTXT* pctxt, 
                                  ASN1ConstCharPtr elemName, 
                                  ASN1ConstCharPtr attributes);
/**
 * This function encodes a named value, for example an enumerated value, such
 * as <element><value/></element>.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a value string
 * @param elemName     A pointer to the name of the element.
 * @param attributes   A pointer to the attributes of the element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncNamedValue (ASN1CTXT* pctxt, 
                                ASN1ConstCharPtr value, 
                                ASN1ConstCharPtr elemName, 
                                ASN1ConstCharPtr attributes);

/**
 * This function encodes an unsigned variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.e.
 * @param value        An unsigned INTEGER value to be encoded. The ASNU1INT
 *                       type is set to the C type 'unsigned int' in the
 *                       asn1type.h file. This is assumed to represent a 32-bit
 *                       integer value.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<INTEGER\>) is added.
 *                       If an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncUInt (ASN1CTXT* pctxt, ASN1UINT value, 
                          ASN1ConstCharPtr elemName);

/**
 * This function encodes an unsigned 64-bit variable of the ASN.1 INTEGER type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        An unsigned 64-bit INTEGER value to be encoded. The
 *                       ASN1UINT64 type is set to the C type 'unsigned
 *                       __int64', 'unsigned long long' or 'unsigned long' in
 *                       the asn1type.h file (depends on the used platform and
 *                       the compiler). This is assumed to represent an
 *                       unsigned 64-bit integer value.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (\<INTEGER\>) is added.
 *                       If an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncUInt64 (ASN1CTXT* pctxt, ASN1UINT64 value, 
                            ASN1ConstCharPtr elemName);

/**
 * This function encodes a variable of the BMPString ASN.1 character string
 * type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a structure representing a 16-bit character
 *                       string to encoded. This structure contains a character
 *                       count element and a pointer to an array of 16-bit
 *                       character elements represented as 16-bit short
 *                       integers.
 * @param elemName     A pointer to a name of an element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncBMPStr (ASN1CTXT* pctxt, 
                            const ASN1BMPString* value, 
                            ASN1ConstCharPtr elemName);

/**
 * This function encodes a variable of the ASN.1 Universal character string.
 * This differs from the encode routines for the character strings previously
 * described in that the Universal string type is based o te 32-bit characters.
 * A 32-bit character string is modeled using an array of unsigned integers.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a structure representing a 16-bit character
 *                       string to encoded. This structure contains a character
 *                       count element and a pointer to an array of 32-bit
 *                       character elements represented as 32-bit short
 *                       integers.
 * @param elemName     A pointer to a name of an element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncUnivStr (ASN1CTXT* pctxt, 
                             const ASN1UniversalString* value, 
                             ASN1ConstCharPtr elemName);

/**
 * This function encodes a variable of the ASN.1 character string types that
 * are based on 16-bit character sets and are represented as null-terminated
 * Unicode string. This includes IA5String, VisibleString, PrintableString,
 * NumericString, and BMPString.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a null-terminated 16-bit string to be
 *                       encoded.
 * @param nchars       The number of characters to be encoded.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncUniCharData (ASN1CTXT* pctxt, 
                                 ASN1Const16BitCharPtr value, 
                                 ASN1UINT nchars);

/** 
 * This function encodes a variable one of the ASN.1 character string types
 * that are based on 16-bit character sets and are represented as
 * null-terminated Unicode strings. This includes IA5String, VisibleString,
 * PrintableString, NumericString, Adn BMPString.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a null-terminated 16-bit character string
 *                       to be encoded.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If an empty string
 *                       is passed (""), no element tag is added to the encoded
 *                       value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncUniCharStr (ASN1CTXT* pctxt, 
                                ASN116BITCHAR* value, 
                                ASN1ConstCharPtr elemName);

/** 
 * This function encodes a variable of the old (pre-1994) ASN.1 ANY type or
 * other elements defined in the later standards to be Open Types (for example,
 * a variable type declaration in a CLASS construct as defined n X.681). A
 * variable of this is considered to be a previously encoded ASN.1 message
 * component.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param nocts        The number of octets (bytes) within the OCTET STRING to
 *                       be encoded.
 * @param data         A pointer to an OCTET STRING containing an encoded ASN.1
 *                       message component.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (<REAL>) is added. If
 *                       an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncOpenType (ASN1CTXT* pctxt, ASN1UINT nocts, 
                              ASN1ConstOctetPtr data, 
                              ASN1ConstCharPtr elemName);

/** 
 * This function encodes an ASN.1 NULL placeholder. In XER the NULL value is
 * represented as an empty element, such as <null/>.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If a null or empty
 *                       string is passes (""), no element tag is added to the
 *                       encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerEncNull (ASN1CTXT* pctxt, ASN1ConstCharPtr elemName);


EXTERNXER int xerEncXmlCharData (ASN1CTXT* pctxt, 
                                 ASN1Const XMLCHAR* pvalue,
                                 int length);
/** @} xerencruntime */

/* utility functions */
/** @defgroup xerutilruntime XER C Utility Functions.
 * @{
 *
 * The XER utility functions are common routines used by both XER encode and
 * decode functions.
 */


/**
 * This function is used to compare two strings: the first is represented as a
 * 16-bit character null-terminated string and the second is represented as an
 * 8-bit standard null-terminated string.
 *
 * @param text1        A pointer to a 16-bit character null-terminated string.
 * @param text2        A pointer to an 8-bit character null-terminated string.
 * @return             The result of the comparison: TRUE, if strings match,
 *                       otherwise FALSE.
 */
EXTERNXER ASN1BOOL xerCmpText (ASN1Const XMLCHAR* text1, ASN1ConstCharPtr text2);


EXTERNXER int xerCopyText (ASN1CTXT* pctxt, ASN1ConstCharPtr text);

/**
 * This function returns the length of a 16-bit character null-terminated
 * string.
 *
 * @param text         A pointer to a 16-bit character null-terminated string.
 */
EXTERNXER int xerTextLength (ASN1Const XMLCHAR* text);

/**
 * This function converts a 16-bit character string to a standard
 * null-terminated C-string. This function will allocate dynamic memory to
 * store the decoded result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param text         A pointer to a 16-bit character null-terminated string.
 * @return             The resulting C string. The memory is allocated by using
 *                       rtMemAlloc function.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNXER ASN1ConstCharPtr xerTextToCStr (ASN1CTXT* pctxt, 
                                          ASN1Const XMLCHAR* text);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function is used to get the encoded message length.
 *
 * @param pctxt        A pointer to a context structure.
 * @return             The length of a message in the buffer.
 */
EXTERNXER int xerGetMsgLen (ASN1CTXT* pctxt);

/**
 * This function is used to obtain a pointer to the start of an encoded
 * message. This function is called after a complier generated encode function
 * to get the pointer to the start of the encoded message. It is normally used
 * when dynamic encoding is specified because the message pointer is not known
 * until the encoding is complete. If static encoding is used, the message
 * starts at the beginning of the specified buffer and the xerGetMsgLen
 * function can be used to obtain the length of the message.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @return             A pointer to the beginning of the encoded message.
 */
EXTERNXER ASN1OCTET* xerGetMsgPtr (ASN1CTXT* pctxt);

EXTERNXER int xerGetElemIdx (ASN1Const XMLCHAR* elemName, 
                             XerElemInfo* pElemInfo, int numElems);

EXTERNXER int xerGetSeqElemIdx (ASN1Const XMLCHAR* elemName, 
                                XerElemInfo* pElemInfo, 
                                int numElems, int startIndex);

EXTERNXER int xerFinalizeMemBuf (ASN1MemBuf* pMemBuf);

/**
 * This function loads the XML SAX library using its name. By default, XER
 * runtime library looks for XML SAX library with name 'expat' (or, in
 * Unix/Linux 'libexpat'). The file name of Windows' version is expat.dll, for
 * Unix/Linux - libexpat.so or libexpat.sl. For better portability, do not add
 * an extension to the file name as a parameter of this function.
 *
 * @param name         Name of a XML SAX library.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerLoadXmlLibrary (const char* name);

/**
 * This function unloads the XML SAX library using its name.
 *
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xerUnloadXmlLibrary (void);

/** @} xerutilruntime */


/* XML decode functions */

/** @defgroup xmldecruntime XML C Decode Functions.
 * @{
 *
 * XML runtime library decode functions handle the decoding of the primitive
 * ASN.1 data types and length variables. Calls to these functions are
 * assembled in the C source code generated by the ASN1C complier to decode
 * complex ASN.1 structures. These functions are also directly callable from
 * within a user's application program if the need to decode a primitive data
 * item exists.
 */

/**
 * This function will decode a variable of the ASN.1 BIT STRING type into a
 * static memory structure. This function call is generated by ASN1C to decode
 * a sized bit production.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a variable to receive the decoded bit
 *                       string. This is assumed to be a static array large
 *                       enough to hold the number of octets specified in the
 *                       bufsize input parameter.
 * @param pnbits       A pointer to an integer value to receive the decoded
 *                       number of bits.
 * @param bufsize      An integer variable containing the size (in octets) of
 *                       the sized ASN.1 bit string. An error will occur if the
 *                       number of octets in the decoded string is larger than
 *                       this value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
/*EXTERNXER int xerDecBitStr (ASN1CTXT* pctxt, ASN1OCTET* pvalue, 
                            ASN1UINT* pnbits, ASN1INT bufsize);*/

/**
 * This function decodes a variable of the ASN.1 BOOLEAN type.
 *
 * @param pctxt        Pointer to context block structure.
 * @param pvalue       Pointer to a variable to receive the decoded BOOLEAN
 *                       value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecBool (ASN1CTXT* pctxt, ASN1BOOL *pvalue);

/**
 * This function will decode a variable of the ASN.1 BIT STRING type. This
 * function will allocate dynamic memory to store the decoded result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a dynamic bit string structure to receive
 *                       the decoded bit string. Dynamic memory is allocated to
 *                       hold the string using the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecDynBitStr (ASN1CTXT* pctxt, ASN1DynBitStr* pvalue);

/** 
 * This function will decode a variable of the ASN.1 OCTET STRING type. This
 * function will allocate dynamic memory to store the decoded result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a dynamic bit string structure to receive
 *                       the decoded bit string. Dynamic memory is allocated to
 *                       hold the string using the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecDynOctStr (ASN1CTXT* pctxt, ASN1DynOctStr* pvalue);

/** 
 * This function will decode a variable of the ASN.1 GeneralizedTime type. This
 * function performs conversion between XML format of dateTime into the ASN.1
 * format. This function will allocate dynamic memory to store the result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param outdata      A pointer to a character string pointer variable to
 *                       receive the decoded string. The string as stored as a
 *                       standard null-terminated C string. Memory is allocated
 *                       for the string by the rtMemAlloc function. It will
 *                       contain time in ASN.1 format.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecGeneralizedTime (ASN1CTXT* pctxt, 
                                     ASN1ConstCharPtr* outdata);

/**
 * This function will decode a list of identifiers into the array of octets.
 * Identifiers should represent named bits value for BIT STRING. This function
 * call is generated by ASN1C to decode a named bit production.
 *
 * @param pMemBuf      Pointer to the destination memory buffer.
 * @param pData        A pointer to a variable to receive the decoded bit
 *                       string. This is assumed to be a static array large
 *                       enough to hold the number of octets specified in the
 *                       bufsize input parameter.
 * @param dataSize     An integer variable containing the size (in octets) of
 *                       the sized ASN.1 bit string. An error will occur if the
 *                       number of octets in the decoded string is larger than
 *                       this value.
 * @param pNumbits     A pointer to an integer value to receive the decoded
 *                       number of bits.
 * @param pBitDict     Bits' dictionary to be used to decode each bit. It is an
 *                       array of name-value pairs, represented by an array of
 *                       XmlNamedBitsDict structure.
 * @param chars        XML data to be appended to memory buffer before parsing.
 *                       Could be NULL, if it is final call to this function.
 * @param length       Number of characters in XML data to be appended to
 *                       memory buffer before parsing. Could be 0, if it is
 *                       final call to this function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecNamedBitStr
  (ASN1MemBuf *pMemBuf, ASN1OCTET* pData, int dataSize, ASN1UINT* pNumbits, 
   ASN1Const XmlNamedBitsDict* pBitDict,
   ASN1Const XMLCHAR* chars, int length);

/**
 * This function will decode a variable of the ASN.1 OCTET STRING type into a
 * static memory structure. This function call is generated by ASN1C to decode
 * a sized octet string production.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a variable to receive the decoded bit
 *                       string. This is assumed to be a static array large
 *                       enough to hold the number of octets specified in the
 *                       bufsize input parameter.
 * @param pnocts       Pointer to an integer value to receive the number of
 *                       octets.
 * @param bufsize      An integer variable containing the size (in octets) of
 *                       the sized ASN.1 octet string. An error will occur if
 *                       the number of octets in the decoded string is larger
 *                       than this value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecOctStr (ASN1CTXT* pctxt, ASN1OCTET* pvalue, 
                            ASN1UINT* pnocts, ASN1INT bufsize);

/** 
 * This function will decode a variable of the ASN.1 32-bit character
 * UniversalString type. This includes the UniversalString type.
 *
 * @param pctxt        A pointer to a context block structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param pvalue       A pointer to a structure variable to receive the decoded
 *                       string. The string as stored as an array of unsigned
 *                       integer characters. Memory is allocated for the string
 *                       by the rtMemAlloc function.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecReal (ASN1CTXT* pctxt, ASN1REAL* pvalue);

/** 
 * This function will decode a variable of the ASN.1 UTCTime type. This
 * function performs conversion between XML format of dateTime into the ASN.1
 * format. This function will allocate dynamic memory to store the result.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param outdata      A pointer to a character string pointer variable to
 *                       receive the decoded string. The string as stored as a
 *                       standard null-terminated C string. Memory is allocated
 *                       for the string by the rtMemAlloc function. It will
 *                       contain time in ASN.1 format.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlDecUTCTime (ASN1CTXT* pctxt, 
                             ASN1ConstCharPtr* outdata);

/** @} xmldecruntime  */


/* XML encode functions */
/** @defgroup xmlencruntime XML C Encode Functions.
 * @{
 *
 * The XML low-level encode functions handle the XML encoding of primitive
 * ASN.1 data types. In most cases XER encoding functions can be used for
 * encoding XML. But there are some differences between XER and XML encodings,
 * described in X.693 and X.694. These functions are very similar to XER ones,
 * and only functions provide different encoding are added.
 *

 */
/** 
 * This function encodes a variable of the ASN.1 BIT STRING type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param namedbits    Bits' dictionary to be used to encode each bit. It is an
 *                       array of name-value pairs, represented by an array of
 *                       XmlNamedBitsDict structure.
 * @param noofnamedbits Number of named bits in bits' dictionary.  
 * @param nbits        The number of bits within the bit string to be encoded.
 * @param data         A pointer to an OCTET string containing the bit data to
 *                       be encoded. This string contains bytes having the
 *                       actual bit settings as they are to be encoded in the
 *                       message.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (<BIT_STRING>) is
 *                       added. If an empty string is passed (""), no element
 *                       tag is added to the encoded value.
 * @param outputType   An enumerated type whose value is set to either
 *                       'ASN1BIN' (for binary format) or 'ASN1HEX' (for
 *                       hexadecimal format).
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlEncBitStr (ASN1CTXT* pctxt, 
                            XmlNamedBitsDict *namedbits,
                            ASN1UINT noofnamedbits,
                            ASN1UINT nbits, 
                            ASN1ConstOctetPtr data, 
                            ASN1ConstCharPtr elemName, 
                            ASN1StrType outputType);

EXTERNXER int xmlEncBoolValue (ASN1CTXT* pctxt, ASN1BOOL value);

/** 
 * This function encodes a variable of the ASN.1 BOOLEAN type.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A BOOLEAN value to be encoded. A BOOLEAN is defined as a
 *                       single octet whose value is 0 for False and any other
 *                       value for TRUE.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If the name is null,
 *                       the default name for this type (<BOOLEAN>) is added.
 *                       If an empty string is passed (""), no element tag is
 *                       added to the encoded value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlEncBool (ASN1CTXT* pctxt, ASN1BOOL value, 
                          ASN1ConstCharPtr elemName);

/** 
 * This function encodes an enumeration value for the ASN.1 ENUMERATED and
 * INTEGER types.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        An enumeration identifier to be encoded.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlEncEnum (ASN1CTXT* pctxt, ASN1ConstCharPtr value);

/** 
 * This function encodes a variable of the ASN.1 GeneralizedTime type. It
 * performs conversion from ASN.1 time format into the XML dateTime format.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a null-terminated C character string to be
 *                       encoded. It should contain GeneralizedTime in ASN.1
 *                       format.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If an empty string
 *                       is passed (""), no element tag is added to the encoded
 *                       value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlEncGeneralizedTime(ASN1CTXT *pctxt, 
                            ASN1ConstCharPtr value, 
                            ASN1ConstCharPtr elemName);

/**
 * This function encodes a named value, for example an enumerated value, such
 * as <element>value</element>.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a value string
 * @param elemName     A pointer to the name of the element.
 * @param attributes   A pointer to the attributes of the element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlEncNamedValue (ASN1CTXT* pctxt, 
                                ASN1ConstCharPtr value, 
                                ASN1ConstCharPtr elemName, 
                                ASN1ConstCharPtr attributes);

/** 
 * This function encodes a variable of the REAL data type. This function
 * provides support for the plus-infinity, minus-infinity, NaN and minus zero
 * special real values. Use the rtGetPlusInfinity, rtGetMinusInfinity, rtGetNaN
 * and rtGetMinusZero functions to get these special values.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A value to be encoded. Special real values are encoded
 *                       by using the apropriate functions to set the real
 *                       value to be encoded.
 * @param elemName     A pointer to the name of the element.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlEncReal (ASN1CTXT* pctxt, ASN1REAL value, 
                          ASN1ConstCharPtr elemName);

/** 
 * This function encodes a variable of the ASN.1 UTCTime type. It performs
 * conversion from ASN.1 time format into the XML dateTime format.
 *
 * @param pctxt        A pointer to a context structure. This provides a
 *                       storage area for the function to store all working
 *                       variables that must be maintained between function
 *                       calls.
 * @param value        A pointer to a null-terminated C character string to be
 *                       encoded. It should contain UTCTime in ASN.1 format.
 * @param elemName     This argument specifies the name of the element that is
 *                       wrapped around the encoded value. If an empty string
 *                       is passed (""), no element tag is added to the encoded
 *                       value.
 * @return             Completion status of operation:
 *                       - 0 (ASN_OK) = success,
 *                       - negative return value is error.
 */
EXTERNXER int xmlEncUTCTime(ASN1CTXT *pctxt, 
                            ASN1ConstCharPtr value, 
                            ASN1ConstCharPtr elemName);

/** @} xmlencruntime */

#ifdef __cplusplus
}
#endif

/** @} xerruntime */

#endif
